package server.parser;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import server.parser.node.DisjunctorNode;
import server.parser.node.Node;
import server.parser.node.QuantifiedNode;
import server.parser.node.VariableNode;

public class FormulaUtil {	
	/**
	 * Prueft, ob die Formel, die dem uebergebenen Baum entspricht, range restricted ist. 
	 * Dazu wird die Menge der range restricted Variablen mit der Menge der freien Variablen verglichen. 
	 * Sind sie identisch ( rr(phi) = free(phi) ), so wird true zurueckgeben, ansonsten false.
	 * 
	 * @param formula Baum, der der Formel entspricht, die auf range restricted ueberprueft werden soll.
	 * @return True, wenn die Formel range restricted ist, sonst false.
	 */
	public static boolean rangerestricted( Node formula ) {
		FormulaToSRNF.srnf( formula );
		Set<String> rr = formula.rr();
		Set<String> free = formula.getFreeVariables();
		
		if( rr == null )
			return false;
		else
			return rr.equals(free);
	}
	
	/**
	 * Translates the variable counter to a string that represents a variable.
	 * @param variableCounter
	 * @return
	 */
	private static String variableCounterToName( int variableCounter ) {
		int remainder = variableCounter % 26;
		int base = variableCounter / 26;
		
		String prefix = "";
		if ( base > 0 ) {
			prefix = variableCounterToName( base-1 );
		}
		
		return prefix + ((char) ('A'+remainder));
	}
	
	/**
	 * Returns a variable not used in this node-tree.
	 * It's a very fast function, you may call it some times.
	 * @param formula
	 * @return
	 */
	public static String getUnusedVariable( Node formula ) {
		int counter = formula.getVariableCounter();
		Set<String> vars = formula.getVariables();
		
		String var = null;
		do {
			var = variableCounterToName( counter );
			counter++;
		} while ( vars.contains(var) );
		
		formula.setVariableCounter( counter );
		
		return var;
	}
	
	/**
	 * Ersetzt alle quantifizierten Variablen mit neuen Namen, die noch nicht in der Formel vorkommen.
	 * Dadurch kann es keine freien und gebundenen Variablen mit den gleichen Namen geben und eine Variable
	 * kann nicht von zwei unterschiedlichen Quantoren gebunden werden wie z.B. "EXISTS X ( EXISTS X (armbruch(X)))".
	 * @param formula Eine Formel, bei der die Variablen ersetzt werden sollen.
	 */
	public static void replaceAllQuantifiedVariables( Node formula ) {
		FormulaUtil.replaceAllQuantifiedVariables( formula, formula );
	}
	
	private static void replaceAllQuantifiedVariables( Node formula, Node root) {
		// Ersetzung muss von innen nach aussen durchgefuehrt werden (innere Quantoren zuerst)
		
		if( formula.isQuantifiedNode() ) {
			QuantifiedNode quantifiedNode = (QuantifiedNode) formula;
			// zunaechst rekursiv die Ersetzung in der quantifizierten Formel durchfuehren
			FormulaUtil.replaceAllQuantifiedVariables(quantifiedNode.getQuantifiedFormula(), root);
			
			// Umbennung der quantifizierten Variablen des aktuellen Knotens
			Map<String, String> renameMapping = new HashMap<String, String>();
			for ( VariableNode var : quantifiedNode.getVariableNodes() ) {
				String newName = getUnusedVariable(root);
				renameMapping.put( var.getVariable(), newName );
			}
			formula.renameVariables(renameMapping);
		}
		else {
			// rekursiv absteigen
			for ( Node child : formula.getChildren() ) {
				FormulaUtil.replaceAllQuantifiedVariables( child, root );
			}
		}
	}
	
	/**
	 * Dient als Hilfsmethode, um die Disjunktion von Formeln zu bilden.
	 * @return Die Disjunktion der Elemente der Formelmenge formulas.
	 * @param formulas, die Formelmenge ueber deren Elemente die Disjunktion gebildet werden soll.
	 */
	public static Formula createDisjunction (List<Formula> formulas) {
		
		DisjunctorNode disjunction = new DisjunctorNode();
		
		for ( Formula policyItem : formulas ) {
			disjunction.addChild( policyItem.clone() );
		}
		
		return new Formula( disjunction );
	}
}
