package server.parser;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import exception.DatabaseException;
import exception.UnsupportedFormulaException;

import server.database.sql.Column;
import server.database.sql.SQLDatabase;
import server.parser.Formatter.SQLFormatter;
import server.parser.node.AtomPredicateNode;
import server.parser.node.ConjunctorNode;
import server.parser.node.DisjunctorNode;
import server.parser.node.EqualityNode;
import server.parser.node.ExistentialQuantifiedNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.VariableNode;

public class FormulaToOracleSQL {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.parser");

	private SQLDatabase db = null;
	private SQLFormatter formatter = null;
	private boolean marked = false;
	private String tableSuffix = "";
	
	public FormulaToOracleSQL( SQLDatabase db ) {
		this.db = db;
		this.formatter = new SQLFormatter();
	}
	
	/**
	 * Setzt den Markierungsmodus fuer die Datenbank. Wenn true uebergeben wird, dann werden marker bei Atomen
	 * beachtet. Bitte 
	 * @param value
	 */
	public void setMarkerMode( boolean value ) {
		this.marked = value;
	}
	
	/**
	 * Sets the table suffix used for SELECT queries. Used in marked databases.
	 * @param suffix
	 */
	public void setTableSuffix( String suffix ) {
		this.tableSuffix = suffix;
	}
	
	
	public String translateFormula( Formula formula ) throws UnsupportedFormulaException, DatabaseException {
		// Kopie der Formel erstellen, da Umformungen durchgefuehrt werden
		formula = (Formula)formula.clone();
		
		if( formula instanceof CompletenessSentenceFormula ) {
			return this.translateCompletenessSentence( (CompletenessSentenceFormula)formula );
		}
		else {
			logger.debug("Original: "+formula);
			FormulaToSRNF.srnf(formula);
			logger.debug("SRNF: "+formula);
			
			// Pruefe, ob die Formel range restricted ist
			if( !FormulaUtil.rangerestricted(formula) ) {
				// Fehler: Formel nicht range restricted
				throw new UnsupportedFormulaException("Formula is not range restricted!");
			}
			
			FormulaToRANF.RANF(formula);
			logger.debug("RANF: "+formula);
			FormulaToRANF.modifiedRANF(formula);
			logger.debug("modified RANF: "+formula);
			return this.translateRANFFormula(formula);
		}
	}
	
	private String translateCompletenessSentence( CompletenessSentenceFormula complete ) throws UnsupportedFormulaException, DatabaseException {
		// Vollstaendigkeitssatz -> genereller Aufbau ist bekannt
		// Umwandlung optimieren, weil die Umwandlung in SRNF, RANF, mRANF, SQL und die
		// Auswertung des generierten Querys sonst untragbar lange dauert.
		Set<String> freeVars = complete.getInteraction().getFreeVariables();
		Set<Map<String, String>> substitutions = complete.getSubstitutions();
		
		// Zielformel:
		// SELECT 'true' AS BOOLEANVALUE FROM DUAL WHERE NOT EXISTS ( 
		//			SELECT * FROM ( sql(interaction) ) WHERE (freeVar_1, ..., freeVar_n) NOT IN ((freeVar_1_1, ..., freeVar_n_1), ..., (freeVar_1_k, ..., freeVar_n_k))
		//		)
		
		String where_clause;
		if( ! complete.getSubstitutions().isEmpty() ) {
			// Resultat einschraenken (Kombinationen bis Element k (einschl.) herausfiltern)
			// -> WHERE-Klausel aufbauen (s.o. Zielformel)
					
			// linke Expression List aufbauen (freie Variablen)
			String left_expList = "(";
			String sep = "";
			for( String freeVar : freeVars ) {
				left_expList += sep + freeVar;
				sep = ",";
			}
			left_expList += ")";
			
			// rechte Expression List aufbauen (auszuschliessende Substitutionen der freien Variablen)
			String right_expList = "(";
			sep = "";
			for( Map<String, String> substitution : substitutions ) {
				right_expList += sep + "(";
				String innerSep = "";
				for( String freeVar : freeVars ) {
					right_expList += innerSep + "'" + substitution.get(freeVar) + "'";
					innerSep = ",";
				}
				right_expList += ")";
				sep = ",";
			}
			right_expList += ")";
		
			where_clause = " WHERE " + left_expList + " NOT IN " + right_expList;
		}
		else {
			// k=0: keine Elemente zum ausschliessen vorhanden -> leere Where-Clause
			where_clause = "";
		}
		
		// Formel in SQL umwandeln
		Formula interaction = (Formula)complete.getInteraction().clone();
		FormulaToSRNF.srnf(interaction);
		String interactionToSQL = this.translateRANFFormula( interaction );
		
		// finalen Query zusammensetzen
		return "SELECT 'true' AS BOOLEANVALUE FROM DUAL WHERE NOT EXISTS (SELECT * FROM (" + interactionToSQL + ")" + where_clause + ")";
	}
	
	/**
	 * Übersetzt eine angegebene (Teil-)Formel in SQL. Je nach Knotentyp wird ein rekursiver Abstieg
	 * bei jeder neuen Teilformel durchgefuehrt. Das Level gibt die aktuelle Rekursionstiefe an und
	 * sollte beim Wurzelknoten mit 0 initialisiert sein.
	 * @param formula
	 * @param level
	 * @return
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 */
	private String translateRANFFormula( Node formula ) throws UnsupportedFormulaException, DatabaseException {
		if ( formula.isAtomPredicateNode() ) {
			return this.translatePredicate( (AtomPredicateNode)formula );
		}
		
		if ( formula.isEqualityNode() ) {
			return this.translateEquality( (EqualityNode)formula );
		}
		
		if ( formula.isConjunctorNode() ) {
			return this.translateConjunction( (ConjunctorNode)formula );
		}
		
		if ( formula.isNegationNode() ) {
			return this.translateNegation( (NegationNode)formula );
		}
		
		if ( formula.isDisjunctorNode() ) {
			return this.translateDisjunction( (DisjunctorNode)formula );
		}
		
		if ( formula.isExistentialQuantifiedNode() ) {
			return this.translateExistentialQuantifier( (ExistentialQuantifiedNode)formula );
		}
		
		if ( formula.isRootNode() ) {
			return this.translateRANFFormula( ((Formula)formula).getRootChild() );
		}
		
		throw new UnsupportedFormulaException( formula.toString() );
	}
	
	/**
	 * Wandelt ein Praedikat der Form P(a,X,b,c,Y,...) in folgendes SQL-Satement um:
	 * SELECT c2 AS X, c5 AS Y FROM P
	 * WHERE c1 = a AND c3 = b AND c4 = c
	 * 
	 * Die c1, ..., cn sind die im Schema vereinbarten Spaltennamen fuer die Tabelle P.
	 * 
	 * Wenn die Menge der Variablen leer ist, dann wuerder auch der SELECT-Teil leer sein.
	 * Deshalb wird dann stattdessen ein SELECT 'true' AS BOOLEANVALUE ausgefuehrt.
	 * 
	 * @param predicate
	 * @return
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException 
	 * @throws DatabaseException 
	 */
	private String translatePredicate( AtomPredicateNode predicate ) throws UnsupportedFormulaException, DatabaseException {
		String select = "";
		String where = "";
		
		// Zuerst den Aufbau der Tabelle aus der Datenbank auslesen und mit dem Praedikat vergleichen.
		List<Column> columns = db.getColumns( predicate.getRelationname() );

		// Die Anzahl der Kinder ist um 1 groesser, weil auch der Name des Praedikats ein Kind ist.
		if ( columns.size() != predicate.getChildrenNumber()-1 ) {
			throw new UnsupportedFormulaException( predicate.toString() );
		}
		
		// SELECT aufbaun.
		select = "SELECT ";
		
		if ( predicate.getVariables().isEmpty() ) {
			// Wenn es keine Variablen gibt, dann kann die Anfrage mit wahr/falsch beantwortet werden.
			select += "'true' AS BOOLEANVALUE";
		}
		
		// Wenn es Variablen gibt, dann wollen wir deren Werte.
		String selectSep = "";
		String whereSep = " WHERE ";

		Iterator<Column> columnIter = columns.iterator();
		for ( Node child : predicate.getChildren() ) {
			if ( child.isVariableNode() ) {
				// Variable gefunden.
				// Dieser Fall trifft nicht mehr zu, wenn der obige Fall mit BOOLEANVALUE zugetroffen ist.
				select += selectSep + columnIter.next().getName() +" AS "+ ((VariableNode)child).getVariable();
				selectSep = ",";
			} else if ( child.isConstantNode() ) {
				// Konstante gefunden.
				// Dieser Fall kann bei beiden Varianten zutreffen.
				//where += whereSep + "LOWER("+ columns[i-1] +") = LOWER('"+ ((ConstantNode)child).getConstant() +"')";
				
				where += whereSep + columnIter.next().getName() +"="+child.toString( this.formatter );
				whereSep = " AND ";
			}
		}
		
		// Wenn es sich bei der Datenbank um eine DB mit temporaeren Tabellen handelt, die markierte Atome hat,
		// dann nur diejenigen Tupel auswaehlen, die nicht mit 'r' (remove) und 'l' (leave) markiert sind.
		if ( this.marked ) {
			where += whereSep + "Marker<>'r' AND Marker<>'l'";
			whereSep = " AND ";
		}

		return select +" FROM "+ predicate.getRelationname() + this.tableSuffix + where;
	}
	
	/**
	 * Uebersetzt Gleichheit (X = Y, X = a, a = X, a = b) und Ungleichheit (X != Y, X != a, a != X, a != b)
	 * außerhalb von Konjunktionen (z.B. Disjunktionen) in einen Wahrheitswert.
	 * Dabei wird von einer unendlichen Domaene ausgegangen und nur die Faelle mit zwei Konstanten koennen
	 * false werden.
	 * @param equality
	 * @return
	 */
	private String translateEquality( EqualityNode equality ) {
		if ( equality.getEquals() ) {
			// Gleichheit
			if ( equality.getLeftOperand().isConstantNode() && equality.getRightOperand().isConstantNode() ) {
				// a = b
				return this.createSQLBoolean( equality.getLeftOperand().equals(equality.getRightOperand()) );
			}
			
			// a = X, X = a: Fuer unendliche Domaene existiert immer ein Element, was gleich a ist.
			// X = Y: Fuer X und Y einfach den gleichen Wert einsetzen => auch immer true.
			return this.createSQLBoolean( true );
		} else {
			// Ungleichheit
			if ( equality.getLeftOperand().isConstantNode() && equality.getRightOperand().isConstantNode() ) {
				// a != b (true), a != a (false)
				return this.createSQLBoolean( !equality.getLeftOperand().equals(equality.getRightOperand()) );
			}
			
			if ( equality.getLeftOperand().isVariableNode() && equality.getRightOperand().isVariableNode() ) {
				// X != X (false) oder X != Y (true, einfach verschiedene Wert einsetzen)
				return this.createSQLBoolean( !equality.getLeftOperand().equals(equality.getRightOperand()) );
			}

			// a != X, X != a: Fuer unendliche Domaene existiert immer ein Element, was ungleich a ist.
			return this.createSQLBoolean( true );
		}
	}
	
	private String translateConjunction( ConjunctorNode and ) throws UnsupportedFormulaException, DatabaseException {
		// Binaere Konjunktion: psi AND xi
		if ( and.getChildrenNumber() != 2 ) {
			throw new UnsupportedFormulaException( and.toString() );
		}
		
		Node psi = null;
		Node xi = null;
		
		if ( !and.getChild(0).isNegationNode() & !and.getChild(0).isEqualityNode() ) {
			psi = and.getChild(0);		
			xi = and.getChild(1);
		}
		else if ( !and.getChild(1).isNegationNode() & !and.getChild(1).isEqualityNode() ) {
			psi = and.getChild(1);
			xi = and.getChild(0);
		}
		// Formel kann nicht ausgewertet werden, da
		// 1) beide Konjunkte sind Negationen
		// 2) beide Konjunkte sind Gleichheiten
		// 3) ein Konjunkt ist Negation, das andere Konjukt Gleichheit
		else {
			throw new UnsupportedFormulaException( and.toString() );
		}
		
		// Rekursive Uebersetzung
		String translatedPsi = this.translateRANFFormula( psi );
		
		// Zuerst Faelle betrachten, bei dem xi eine Gleichheit oder Ungleichheit ist.
		if ( xi.isEqualityNode() ) {
			EqualityNode equality = (EqualityNode)xi;
			if ( equality.getEquals() ) {
				// xi (rechter Junctor) ist Gleichheit

				VariableNode var = null;
				if ( equality.getLeftOperand().isConstantNode() && equality.getRightOperand().isVariableNode() )
					var = (VariableNode)equality.getRightOperand();
				if ( equality.getLeftOperand().isVariableNode() && equality.getRightOperand().isConstantNode() )
					var = (VariableNode)equality.getLeftOperand();
				
				if ( var != null ) {
					// Es gibt genau eine Konstante in der Gleichheit.
					
					// Unterscheide ob psi Variable enthaelt (gebunden vorkommt) oder nicht.
					if ( psi.getBoundVariables().contains(var.getVariable()) || !psi.getFreeVariables().contains(var.getVariable()) ) {
						// Die Variable der Gleichheit kommt nicht frei oder gar nicht vor => Spalte existiert im Ergebnis
						// der Anfrage nicht. Dann ist die rechte Seite aber trivialerweise immer true
						// und wir koennen einfach das Ergebnis der linken Formel zurueckgeben.
						return translatedPsi;
					} else {
						// Linke Formel enthaelt die Variable frei => Ergebnis fuer diese freie Variable
						// durch die Gleichheit einschraenken.
						return "SELECT * FROM ("+translatedPsi+") WHERE "+equality.toString( this.formatter );
					}
				}
				
				if ( equality.getLeftOperand().isConstantNode() && equality.getRightOperand().isConstantNode() ) {
					return "SELECT * FROM ("+translatedPsi+") WHERE "+equality.toString( this.formatter );
				}
				
				if ( equality.getLeftOperand().equals(equality.getRightOperand()) ) {
					// xi ist X = X
					return translatedPsi;
				}
				
				// xi ist X = Y
				VariableNode x = (VariableNode)equality.getLeftOperand();
				VariableNode y = (VariableNode)equality.getRightOperand();
				
				Set<String> free = psi.getFreeVariables();
				
				if ( free.contains(x.getVariable()) && free.contains(y.getVariable())) {
					// {X,Y} Teilmenge free(psi).
					return "SELECT * FROM ("+translatedPsi+") WHERE "+equality.toString( this.formatter );
				}
				
				if ( free.contains(x.getVariable()) && !free.contains(y.getVariable())) {
					// x in free(psi) und y not in free(psi).
					return this.createAlgebraJoin( translatedPsi, this.useRenamingFunction(x.getVariable(), y.getVariable(), translatedPsi) );
				}
				
				if ( !free.contains(x.getVariable()) && free.contains(y.getVariable())) {
					// x not in free(psi) und y in free(psi).
					return this.createAlgebraJoin( translatedPsi, this.useRenamingFunction(y.getVariable(), x.getVariable(), translatedPsi) );
				}
				
				
				// Formel ist nicht range-restricted, wenn man hier ankommt..
				throw new UnsupportedFormulaException( and.toString() );
			}
			
			// xi ist Ungleichheit (X != Y, X != X, a != b, a != a).
			return "SELECT * FROM ("+translatedPsi+") WHERE "+equality.toString( this.formatter );
		}
		
		// Fall xi ist eine Negation betrachten.
		if ( xi.isNegationNode() ) {
			Node xiApostrophe = ((NegationNode)xi).getNegatedFormula();
			
			Set<String> freePsi = psi.getFreeVariables();
			
			if ( freePsi.equals(xiApostrophe.getFreeVariables()) ) {
				// free(xi') = free(psi).
				return this.createSubstraction(psi.getFreeVariables(), translatedPsi, this.translateRANFFormula(xiApostrophe) );
			}
			
			if ( freePsi.containsAll(xiApostrophe.getFreeVariables()) ) {
				// free(xi') echte Teilmenge free(psi) (wegen = im Fall oben abgedeckt).
				String join = this.createAlgebraJoin( translatedPsi, this.translateRANFFormula(xiApostrophe) );
				Set<String> vars = psi.getFreeVariables();
				if ( xiApostrophe.isClosedFormula() ) {
					vars.add( "'true'" );
				}
				return this.createSubstraction(vars, translatedPsi, join );
			}
			
			// Formel ist nicht range-restricted, wenn man hier ankommt..
			throw new UnsupportedFormulaException( and.toString() );
		}
		
		// Ansonsten einfach normalen Algebra-Join (NATURAL INNER JOIN).
		return this.createAlgebraJoin( translatedPsi, this.translateRANFFormula(xi) );
	}
	
	/**
	 * Wandelt eine Negation der Form NOT (phi) in folgendes SQL-Statement um:
	 * SELECT 'true' AS BOOLEANVALUE WHERE NOT EXISTS (phi)
	 * 
	 * Das (phi) wird durch rekursiven Aufruf der Methode translateFormula ersetzt.
	 * Als Ergebnis kommt entweder eine leere Tabelle oder ein 'true' in der Spalte
	 * BOOLEANVALUE zurueck.
	 * 
	 * @param negation
	 * @return
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException 
	 */
	private String translateNegation( NegationNode negation ) throws UnsupportedFormulaException, DatabaseException {
		return "SELECT 'true' AS BOOLEANVALUE FROM DUAL WHERE NOT EXISTS ("+ this.translateRANFFormula( negation.getNegatedFormula() ) +")";
	}
	
	/**
	 * Wandelt eine Disjunction der Form phi_1 v phi_2 v ... v phi_n in folgendes SQL-Statement um:
	 * SELECT * FROM (phi_1)
	 * NATURAL FULL OUTER JOIN (phi_2)
	 * ...
	 * NATURAL FULL OUTER JOIN (phi_n)
	 * 
	 * Die (phi_i) werden durch rekursiven Aufruf der Methode translateFormula ersetzt.
	 */
	private String translateDisjunction( DisjunctorNode or ) throws UnsupportedFormulaException, DatabaseException {
		List<Node> children = or.getChildren();
		String join = "";
		
		// Baue JOIN auf.
		for ( int i = 0; i < children.size(); i++ ) {
			// Das erste Kind kommt nicht in den JOIN sondern in FROM
			if ( i > 0 ) {
				join += "NATURAL FULL OUTER JOIN ("+ this.translateRANFFormula(children.get(i)) +") ";
			}
		}
		
		String sql = "SELECT * FROM ("+ this.translateRANFFormula(children.get(0)) +") "+ join;
				
		return sql;
	}
	
	private String translateExistentialQuantifier( ExistentialQuantifiedNode exists ) throws UnsupportedFormulaException, DatabaseException {
		if ( exists.getQuantifiedVariables().containsAll(exists.getQuantifiedFormula().getFreeVariables()) ) {
			// Query ist komplett geschlossen => nur Wahrheitswert zurueckgeben.
			return "SELECT 'true' AS BOOLEANVALUE FROM ("+this.translateRANFFormula(exists.getQuantifiedFormula())+")";
		} else {
			// Es gibt noch offene Variablen => die quantifizierten ausblenden.
			String variables = "";
			String sep = "";
			
			for ( String var : exists.getFreeVariables() ) {
				variables += sep + var;
				sep = ",";
			}
			// es kommen in der Anfrage nur quantifizierte Varialben vor, die unrelevant sind
			// entsprechend werden keine Spalten projeziert -> SELECT * FROM ...
			if ( variables.isEmpty() ) {
				return "SELECT * FROM ("+this.translateRANFFormula(exists.getQuantifiedFormula())+")";				
			}

			return "SELECT "+variables+" FROM ("+this.translateRANFFormula(exists.getQuantifiedFormula())+")";

		}
	}
	
	/**
	 * Wandelt eine Konjunktion der Form phi_1 AND phi_2 AND ... AND phi_n in folgendes SQL-Statement um:
	 * SELECT * FROM (phi_1)
	 * NATURAL INNER JOIN (phi_2)
	 * ...
	 * NATURAL INNER JOIN (phi_n)
	 * 
	 * Die (phi_i) werden durch rekursiven Aufruf der Methode translateFormula ersetzt.
	 * 
	 * @param and
	 * @return
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException 
	 */
	private String createAlgebraJoin( String query1, String query2 ) throws UnsupportedFormulaException, DatabaseException {
		return "SELECT * FROM ("+query1+") NATURAL INNER JOIN ("+query2+")"; 
	}
	
	private String createSQLBoolean( boolean bool ) {
		if ( bool ) {
			return "SELECT 'true' AS BOOLEANVALUE FROM DUAL";
		} else {
			return "SELECT 'true' AS BOOLEANVALUE FROM DUAL WHERE 'a'='b'";
		}
	}
	
	private String useRenamingFunction( String renameFrom, String renameTo, String query ) {
		return "SELECT "+renameFrom+" AS "+renameTo+" FROM ("+query+")";
	}
	
	private String createSubstraction( Set<String> variables, String left, String right ) throws UnsupportedFormulaException, DatabaseException {
		String freeVars = "";
		
		if( variables.isEmpty() ) {
			// keine Variablen vorhanden -> SELECT part von left und right enthalten: 'true' AS BOOLEANVALUE"
			
			return "SELECT * FROM ("+left+") WHERE ('true') NOT IN ("+right+")";
		}
		else {
			String freeVarsSelect = "";
			String sep = "";
			for ( String freeVar : variables ) {
				freeVars += sep + freeVar;
				if( freeVar.equals("'true'") )
					freeVarsSelect += sep + freeVar + " AS BOOLEANVALUE";
				else
					freeVarsSelect += sep + freeVar;
				sep = ",";
			}
			
			return "SELECT * FROM ("+left+") WHERE ("+freeVars+") NOT IN (SELECT "+freeVarsSelect+" FROM ("+right+"))";
		}
	}
}
