package server.database.sql;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;
import java.util.regex.PatternSyntaxException;

import org.apache.log4j.Logger;

import exception.DatabaseException;

import server.database.schema.SchemaColumn;
import server.database.schema.TableSchema;
import server.util.Tuple;

/**
 * Resembles an SQL SELECT statement.
 * Please see the note for method chaining in the description
 * of SQLBaseExpression.
 *
 * @see SQLBaseExpression
 * @see SQLWhereExpression
 */
public class SQLSelect extends SQLWhereExpression<SQLSelect> {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");

	// Currently used to specify which database backend to use.
	// FIXME: move to class hierarchy,
	private final static String driver = "oracle";

	private List<String> select;
	private boolean distinct;
	private List<Tuple<String,Boolean>> orderBy;
	private List<String> groupBy;
	private List<Tuple<String,List<Tuple<String,String>>>> join;
	private int limit;
	
	/**
	 * Creates a new SQL SELECT expression.
	 * @param db Database connection. Must already be established and usable.
	 */
	public SQLSelect( SQLDatabase db ) {
		super( db );
		
		this.select = new Vector<String>();
		this.distinct = false;
		this.orderBy = new Vector<Tuple<String,Boolean>>();
		this.groupBy = new Vector<String>();
		this.join = new LinkedList<Tuple<String,List<Tuple<String,String>>>>();
		this.limit = -1;
	}
	
	/**
	 * Returns the List of columns whose data will be in the result set of this SELECT statement.
	 * @return List of columns.
	 * @see #select(SchemaColumn)
	 * @see #select(String)
	 */
	public List<String> getSelect() {
		return this.select;
	}
	
	/**
	 * Selects a column whose data should be retrieved. Can be called
	 * multiple times to select more that one column.
	 * @param column Column that will be selected in the query.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getSelect()
	 * @see #select(String)
	 */
	public SQLSelect select( SchemaColumn column ) {
		this.select.add( column.getFullName() );
		
		return this;
	}
	
	/**
	 * Selects one or multiple column(s) whose data should be retrieved.
	 * Use comma as separator for multiple columns.
	 * Can also be called multiple times to select more that one column.
	 * Try to use {@link #select(SchemaColumn)} if possible.
	 * @param column Column that will be selected in the query.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getSelect()
	 * @see #select(SchemaColumn)
	 */
	public SQLSelect select( String column ) {
		String[] split = null;
		
		try {
			split = column.split( "," );
			
			for ( String col : split ) {
				this.select.add( col );
			}
		} catch ( PatternSyntaxException exception ) {
			// This one should be impossible. The regex pattern is always valid.
			logger.fatal("Invalid regex pattern in select(). This is impossible. Check your code.", exception);
			return this;
		}
			
		return this;
	}
	
	/**
	 * Returns whether the results should contain duplicates or not.
	 * @return True, if the result will NOT contain any duplicates, false otherwise.
	 * @see #distinct()
	 */
	public boolean isDistinct() {
		return this.distinct;
	}
	
	/**
	 * Disables any duplicates in the result. See {@link #isDistinct()}.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #isDistinct()
	 */
	public SQLSelect distinct() {
		this.distinct = true;
		
		return this;
	}
	
	/**
	 * Specifies the table to select from.
	 * Calls {@link SQLBaseExpression#table(TableSchema)}.
	 * @param table Table used for SELECT statement.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see SQLBaseExpression#table(TableSchema)
	 * @see SQLBaseExpression#table(String)
	 * @see #from(String)
	 */
	public SQLSelect from( TableSchema table ) {
		return super.table(table);
	}
	
	/**
	 * Specifies the table to select from.
	 * Calls {@link SQLBaseExpression#table(String)}.
	 * @param table Table used for SELECT statement.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see SQLBaseExpression#table(TableSchema)
	 * @see SQLBaseExpression#table(String)
	 * @see #from(TableSchema)
	 */
	public SQLSelect from( String table ) {
		return super.table(table);
	}
		
	/**
	 * Gets a list of columns and sort order (second tuple value false = ascending, true = descending)
	 * for the result.
	 * @return List containing tuples of columns and a sort order.
	 * @see #order_by(SchemaColumn)
	 * @see #order_by(String)
	 * @see #order_by(SchemaColumn, boolean)
	 * @see #order_by(String, boolean)
	 */
	public List<Tuple<String,Boolean>> getOrderBy() {
		return this.orderBy;
	}
	
	/**
	 * Adds a column that should be used for sorting the output in ascending order.
	 * Can be called multiple times.
	 * @param column Column that will be used to sort the result in ascending order.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getOrderBy()
	 * @see #order_by(String)
	 * @see #order_by(SchemaColumn, boolean)
	 * @see #order_by(String, boolean)
	 */
	public SQLSelect order_by( SchemaColumn column ) {
		return this.order_by( column, false );
	}
	
	/**
	 * Adds a column that should be used for sorting the output. The sort order
	 * is determined by the parameter desc.
	 * @param column Column that will be used to sort the result.
	 * @param desc True if the result should be sorted in descending order, false for ascending order.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getOrderBy()
	 * @see #order_by(SchemaColumn)
	 * @see #order_by(String)
	 * @see #order_by(String, boolean)
	 */
	public SQLSelect order_by( SchemaColumn column, boolean desc ) {
		this.orderBy.add( new Tuple<String,Boolean>(column.getFullName(), desc) );
		
		return this;
	}
	
	/**
	 * Adds a column that should be used for sorting the output in ascending order.
	 * Can be called multiple times.
	 * Try to use {@link #order_by(SchemaColumn)} if possible.
	 * @param column Column that will be used to sort the result in ascending order.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getOrderBy()
	 * @see #order_by(SchemaColumn)
	 * @see #order_by(SchemaColumn, boolean)
	 * @see #order_by(String, boolean)
	 */
	public SQLSelect order_by( String column ) {
		return this.order_by( column, false );
	}
	
	/**
	 * Adds a column that should be used for sorting the output. The sort order
	 * is determined by the parameter desc.
	 * Try to use {@link #order_by(SchemaColumn, boolean)} if possible.
	 * @param column Column that will be used to sort the result.
	 * @param desc True if the result should be sorted in descending order, false for ascending order.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getOrderBy()
	 * @see #order_by(SchemaColumn)
	 * @see #order_by(String)
	 * @see #order_by(SchemaColumn, boolean)
	 */
	public SQLSelect order_by( String column, boolean desc ) {
		this.orderBy.add( new Tuple<String,Boolean>(column, desc) );
		
		return this;
	}
	
	/**
	 * Returns the list of columns that will be used for grouping the result.
	 * @return List of columns used for grouping the result.
	 * @see #group_by(SchemaColumn)
	 * @see #group_by(String)
	 */
	public List<String> getGroupBy() {
		return this.groupBy;
	}
	
	/**
	 * Adds a column that will be used for grouping the result.
	 * @param column Name of the column that will be used to group the result.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getGroupBy()
	 * @see #group_by(String)
	 */
	public SQLSelect group_by( SchemaColumn column ) {
		this.groupBy.add( column.getFullName() );
		
		return this;
	}
	
	/**
	 * Adds a column that will be used for grouping the result.
	 * Try to use {@link #group_by(SchemaColumn)} if possible.
	 * @param column Name of the column that will be used to group the result.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #group_by(SchemaColumn)
	 * @see #getGroupBy()
	 */
	public SQLSelect group_by( String column ) {
		this.groupBy.add( column );
		
		return this;
	}
	
	/**
	 * Returns the list of tuples containing table names and join conditions that will be used
	 * for generating a result spanning multiple tables.
	 * @return List of tuples containing tables names and join conditions.
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public List<Tuple<String, List<Tuple<String,String>>>> getJoin() {
		return this.join;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * Will perform a NATURAL JOIN.
	 * Try to use {@link #join(TableSchema)} if possible.
	 * @param table Table which will be part of the result.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( String table ) {
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table, null) );
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * Will perform a NATURAL JOIN.
	 * @param table Table which will be part of the result.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( TableSchema table ) {
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table.name, null) );
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * The condition can be null, which will perform a NATURAL JOIN, or two columns, one out
	 * of each table, which will result in an INNER JOIN.
	 * Try to use {@link #join(TableSchema, List)} if possible.
	 * @param table Table which will be part of the result.
	 * @param on Join condition. Can either be null or a list of tuples containing two columns, one for each table.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( String table, List<Tuple<String,String>> on ) {
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table, on) );
		
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * The condition can be null, which will perform a NATURAL JOIN, or two columns, one out
	 * of each table, which will result in an INNER JOIN.
	 * @param table Table which will be part of the result.
	 * @param on Join condition. Can either be null or a list of tuples containing two columns, one for each table.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( TableSchema table, List<Tuple<String,String>> on ) {
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table.name, on) );
		
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * Performs an INNER JOIN, so column1 and column2 must be of different tables.
	 * Try to use {@link #join(TableSchema, SchemaColumn, SchemaColumn)} if possible.
	 * @param table Table which will be part of the result.
	 * @param column1 Column of the original table.
	 * @param column2 Column of the joined table.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( String table, String column1, String column2 ) {
		List<Tuple<String,String>> onList = new LinkedList<Tuple<String,String>>();
		onList.add( new Tuple<String,String>(column1, column2) );
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table, onList) );
		
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * Performs an INNER JOIN, so column1 and column2 must be of different tables.
	 * Try to use {@link #join(TableSchema, SchemaColumn, SchemaColumn)} if possible.
	 * @param table Table which will be part of the result.
	 * @param column1 Column of the original table.
	 * @param column2 Column of the joined table.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( TableSchema table, String column1, String column2 ) {
		List<Tuple<String,String>> onList = new LinkedList<Tuple<String,String>>();
		onList.add( new Tuple<String,String>(column1, column2) );
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table.name, onList) );
		
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * Performs an INNER JOIN, so column1 and column2 must be of different tables.
	 * Try to use {@link #join(TableSchema, SchemaColumn, SchemaColumn)} if possible.
	 * @param table Table which will be part of the result.
	 * @param column1 Column of the original table.
	 * @param column2 Column of the joined table.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(TableSchema, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( String table, SchemaColumn column1, SchemaColumn column2 ) {
		List<Tuple<String,String>> onList = new LinkedList<Tuple<String,String>>();
		onList.add( new Tuple<String,String>(column1.getFullName(), column2.getFullName()) );
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table, onList) );
		
		return this;
	}
	
	/**
	 * Adds a new table that will be used for generating a result spanning multiple tables.
	 * Performs an INNER JOIN, so column1 and column2 must be of different tables.
	 * @param table Table which will be part of the result.
	 * @param column1 Column of the original table.
	 * @param column2 Column of the joined table.
	 * @return Current SQLSelect object. Allows method chaining.
	 * @see #getJoin()
	 * @see #join(String)
	 * @see #join(TableSchema)
	 * @see #join(TableSchema, List)
	 * @see #join(String, List)
	 * @see #join(String, String, String)
	 * @see #join(TableSchema, String, String)
	 * @see #join(String, SchemaColumn, SchemaColumn)
	 */
	public SQLSelect join( TableSchema table, SchemaColumn column1, SchemaColumn column2 ) {
		List<Tuple<String,String>> onList = new LinkedList<Tuple<String,String>>();
		onList.add( new Tuple<String,String>(column1.getFullName(), column2.getFullName()) );
		this.join.add( new Tuple<String,List<Tuple<String,String>>>(table.name, onList) );
		
		return this;
	}
	
	/**
	 * Returns the maximum number of rows that will be in the result.
	 * @return Maximum number of result rows.
	 */
	public int getLimit() {
		return this.limit;
	}

	/**
	 * Sets the maximum number of rows that will be in the result to the given value.
	 * @param number Maximum number of rows in the result.
	 * @return Current SQLSelect object. Allows method chaining.
	 */
	public SQLSelect limit( int number ) {
		this.limit = number;
		
		return this;
	}

	
	@Override
	protected String toSQL( boolean bindVars ) throws DatabaseException {
		String result = "";
		String sep = "";
		
		if ( this.getTable() == null ) {
			throw new DatabaseException("No table for SELECT specified.", null);
		}
		
		// SELECT ..
		result = "SELECT ";
		
		if ( this.distinct ) {
			result += "DISTINCT ";
		}
		
		if ( this.select.size() == 0 ) {
			result += "*";
		} else {
			for ( String column : this.select ) {
				result += sep + column;
				sep = ",";
			}
		}
		
		// FROM ..
		result += " FROM "+this.getTable();
		
		// INNER/NATURAL JOIN ..
		for ( Tuple<String,List<Tuple<String,String>>> join : this.join ) {
			if ( join.getSecondElement() == null ) {
				// NATURAL JOIN
				result += " NATURAL JOIN "+join.getFirstElement();
			}
			else
			{
				// INNER JOIN
				result += " INNER JOIN "+join.getFirstElement() + " ON (";
				
				sep = "";
				for ( Tuple<String,String> on : join.getSecondElement() ) {
					result += on.getFirstElement() + " = " + on.getSecondElement();
					sep = ",";
				}
				
				result += ")";
			}
		}
		
		// WHERE ..
		result += this.generateWhereSQL( bindVars );
		
		// GROUP BY
		if ( this.groupBy.size() > 0 ) {
			result += " GROUP BY ";
			
			sep = "";
			for ( String column : this.groupBy) {
				result += sep + column;
				sep = ",";
			}
		}
		
		// ORDER BY
		if ( this.orderBy.size() > 0 ) {
			result += " ORDER BY ";
			
			sep = "";
			for ( Tuple<String,Boolean> column : this.orderBy ) {
				result += sep + column.getFirstElement();
				if ( column.getSecondElement() ) {
					result += " DESC";
				}
				sep = ",";
			}
		}
		
		// LIMIT
		if ( this.limit > 0 ) {
			if ( driver.equalsIgnoreCase("oracle") ) {
				result = "SELECT * FROM ("+result+") WHERE ROWNUM <= "+this.limit;
			} else if ( driver.equalsIgnoreCase("mysql") ) {
				result += " LIMIT "+this.limit;
			}
		}
		
		return result;
	}
	
	
	
	
	
	
	/**
	 * Executes the SELECT statement and returns the result.
	 * @return Result of the SELECT statement.
	 * @throws DatabaseException
	 * @see SQLTable
	 */
	public SQLTable get() throws DatabaseException {
		SQLTable result = null;
		
		// Create SQL statement.
		String sql = this.toSQL( true );
		
		logger.debug("SQL SELECT: " + sql);
		logger.debug("Bind Values: "+this.getWhereBindVariables());
		
		// Replace variables.
		PreparedStatement pstmt = this.getStatement( sql, this.getWhereBindVariables() );

		// Execute SQL statement.
		try {
			ResultSet rs = pstmt.executeQuery();
			result = new SQLTable( rs );
			pstmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Error in the current SQL SELECT statement: "+exception.toString(), exception );
		}
		
		return result;
	}

}
