package server.database.sql;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import server.database.schema.TableSchema;

import exception.DatabaseException;

/**
 * This class must be used as parent class to all SQL expressions.
 * 
 * On method chaining:
 * 
 * Most of the methods setting parts of the SQL expressions return
 * the object on which the method was called. This can be used
 * to build chains of methods, each setting a different value.
 * Doing so allows you to resemble the structure of an SQL expression
 * within your code and make it easier to read.
 * If you don't want to use this feature, just ignore the return
 * values.
 * 
 * Method chaining example:
 * SQLSelect sql = new SQLSelect(db);
 * sql.select("name").table("people").where("age",18,Condition.GREATER);
 * 
 * The last statement is equal to:
 * sql.select("name");
 * sql.table("people");
 * sql.where("age",18,Condition.GREATER);
 * 
 * @param <T> This MUST be the type of the child class. Otherwise method chaining WILL FAIL.
 */
public abstract class SQLBaseExpression<T> {
	
	protected SQLDatabase db;
	private String table;
	
	public SQLBaseExpression( SQLDatabase db ) {
		this.db = db;
	}
	
	/**
	 * Returns the table used for this SQL statement.
	 * @return Table affected by this SQL statement.
	 * @see #table(String)
	 * @see #table(TableSchema)
	 */
	public String getTable() {
		return this.table;
	}
	
	/**
	 * Sets the table used for this SQL statement.
	 * @param table Table affected by this SQL statement.
	 * @return Current SQL expression object. Allows method chaining.
	 * @see #getTable()
	 * @see #table(TableSchema)
	 */
	@SuppressWarnings("unchecked")
	public T table(String table) {
		this.table = table;
		return (T)this;
	}
	
	/**
	 * Sets the table used for this SQL statement.
	 * @param table Table affected by this SQL statement.
	 * @return Current SQL expression object. Allows method chaining.
	 * @see #getTable()
	 * @see #table(String)
	 */
	@SuppressWarnings("unchecked")
	public T table(TableSchema table) {
		this.table = table.name;
		return (T)this;
	}

	/**
	 * Returns the current SQL expression in a human readable form.
	 * This means no variables or other wildcards.
	 * @return String representation of the SQL expression.
	 * @throws DatabaseException 
	 */
	public String toSQL() throws DatabaseException {
		return this.toSQL( false );
	}
	
	/**
	 * Transforms the object to a SQL String. The parameter determines if the result
	 * should be human readable (thus not containing any wildcards like bind variables) or
	 * should be processable by the SQL driver.
	 * @param bindVars True if bind variables should be used, false otherwise.
	 * @return SQL statement as a string.
	 * @throws DatabaseException 
	 */
	protected abstract String toSQL( boolean bindVars ) throws DatabaseException;
	
	/**
	 * Calls {@link #toSQL()}. Returns "(invalid statement)" if an exception occurs.
	 */
	@Override
	public String toString() {
		try {
			return this.toSQL();
		} catch ( DatabaseException ex ) {
			return "(invalid statement)";
		}
	}
	
	/**
	 * Creates an PreparedStatement, which is used in conjunction with bind variables and SQL expressions.
	 * The statement is accepted by the SQL driver.
	 * @param sql String representation of an SQL expression.
	 * @param vars Values of the bind variables occurring in the SQL expression.
	 * @return The completed PreparedStatement, ready to be sended to the database.
	 * @throws DatabaseException Thrown if an SQLException occurred during the process.
	 */
	protected PreparedStatement getStatement( String sql, List<Object> vars ) throws DatabaseException {
		return this.getStatement(sql, vars, null);
	}
	
	/**
	 * Creates an PreparedStatement, which is used in conjunction with bind variables and SQL expressions.
	 * The statement is accepted by the SQL driver.
	 * @param sql String representation of an SQL expression.
	 * @param vars Values of the bind variables occurring in the SQL expression.
	 * @param generatedColumns String array with columns automatically generated by INSERT statements. Can be empty.
	 * @return The completed PreparedStatement, ready to be sended to the database.
	 * @throws DatabaseException Thrown if an SQLException occurred during the process.
	 */
	protected PreparedStatement getStatement( String sql, List<Object> vars, String[] generatedColumns ) throws DatabaseException {
		PreparedStatement pstmt = null;
		
		try {
			if ( generatedColumns != null && generatedColumns.length > 0 ) {
				pstmt = this.db.getDatabaseConnection().prepareStatement( sql, generatedColumns );
			} else {
				pstmt = this.db.getDatabaseConnection().prepareStatement( sql );
			}

			// Set values for bind variables.
			int i = 1;
			for ( Object obj : vars ) {
				pstmt.setObject( i, obj );
				i++;
			}
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Error while replacing bind variables (questionmarks) in the current sql expression.", exception );
		}
		
		return pstmt;
	}
}
