package server.database.sql;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import server.database.DatabaseConfiguration;
import server.database.schema.TableSchema;
import exception.DatabaseException;

import oracle.jdbc.pool.OracleDataSource;

public class OracleSQLDatabase extends SQLDatabase {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database.sql");

	private SQLTable generatedKeys;
	
	/**
	 * Erzeugt eine neue Datenbankverbindung zu einer Oracle-Datenbank.
	 * @throws DatabaseException Wird ausgeloest, wenn keine Datenbankverbindung moeglich ist.
	 */
	public OracleSQLDatabase( DatabaseConfiguration config ) throws DatabaseException {
		super(config);
		
		this.generatedKeys = null;
	}
	
	@Override
	public void connect( DatabaseConfiguration config ) throws SQLException {
		if ( config.getType().equals("oracle") ) {
			OracleDataSource ds = new OracleDataSource();

			logger.info("Connecting to database " + config.getDb() + " with user "+config.getLogin()+" at server "+config.getServer()+":"+config.getPort());
			ds.setDriverType("thin");
			ds.setDatabaseName( config.getDb() );
			ds.setUser( config.getLogin() );
			ds.setPassword( config.getPassword() );
			ds.setServerName( config.getServer() );
			ds.setPortNumber( config.getPort() );
						
			this.connection = ds.getConnection();
	    }
	}

	@Override
	public void commit() throws DatabaseException {
		this.rawQuery("COMMIT");
	}
	
	@Override
	public void rollback() throws DatabaseException {
		this.rawQuery("ROLLBACK");
	}

	@Override
	public void lockExclusive(String table) throws DatabaseException {
		this.rawQuery("LOCK TABLE "+table+" IN EXCLUSIVE MODE");
	}

	@Override
	public void lockShared(String table) throws DatabaseException {
		this.rawQuery("LOCK TABLE "+table+" IN SHARE MODE");
	}

	@Override
	public List<String> getTableNames() throws DatabaseException {
		List<String> result = new ArrayList<String>();
		
		SQLSelect sql = this.newSelect();
		sql.select("table_name").from("user_tables");
		SQLTable table = sql.get();
		
		for ( String[] row : table ) {
			result.add( row[0] );
		}
		
		return result;
	}
	
	/**
	 * Fuehrt ein SQL INSERT Statement aus. Schlaegt die Ausfuehrung des INSERT fehlt,
	 * so wird eine Exception geworfen. Ansonsten kann davon ausgegangen werden, dass
	 * der INSERT erfolgreich ausgefuehrt wurde (silence = everything good).
	 * 
	 * Die einzufuegenden Daten muessen zuvor mit set() hinzugefuegt werden.
	 * Nach jeder Ausfuehrung von insert() wird der Cache fuer die einzufuegenden
	 * Werte zurueckgesetzt.
	 * 
	 * @param table Gibt an in welche Tabelle (Relation) die Daten eingefuegt werden sollen.
	 * @throws DatabaseException
	 */
	@Override
	public int insert( String table ) throws DatabaseException {
		int result = -1;
		
		PreparedStatement pstmt = null;
		try {
			int generatedColumnSize = this.getCurrentQuery().getGeneratedColumns().size();
			if ( generatedColumnSize > 0 ) {
				String[] generatedColumnList = this.getCurrentQuery().getGeneratedColumns().toArray( new String[generatedColumnSize] );
				pstmt = this.connection.prepareStatement( this.getCurrentQuery().getInsertSQL(table, true), generatedColumnList );
			} else {
				pstmt = this.connection.prepareStatement( this.getCurrentQuery().getInsertSQL(table, true) );
			}
			
			int i = 1;
			for ( Object obj : this.getCurrentQuery().getVars() ) {
				pstmt.setObject( i, obj );
				i++;
			}
			
			logger.debug("INSERT SQL:"+this.getCurrentQuery().getInsertSQL(table));
			
			result = pstmt.executeUpdate();
			
			// Automatisch generierte Keys im ResultSet speichern.
			if ( generatedColumnSize > 0 ) {
				//this.generatedKeys = pstmt.getGeneratedKeys();
				this.generatedKeys = new SQLTable( pstmt.getGeneratedKeys() );
			} else {
				this.generatedKeys = new SQLTable( null );
			}
			
			pstmt.close();
			
		} catch( SQLException e ) {
			throw new DatabaseException( "INSERT konnte nicht ausgeführt werden.", e );
		}
		
		// Query neu initialisieren
		this.sql();
		
		return result;
	}
	
	@Override
	public int insert( TableSchema table ) throws DatabaseException {
		return this.insert( table.name );
	}

	/**
	 * Kann nach Auswertung eines INSERT-Ausdruck (siehe insert()) die ggf. automatisch generierten Spaltenwerte abrufen.
	 * Wurden keine Spaltenwerte generiert kann ein leeres ResultSet oder null zurueckgegeben werden.

	 * @return ResultSet, in dem die automatisch generierten Spalten enthalten sind. Kann auch leer oder null sein, wenn kein Wert generiert wurde.
	 */
	public SQLTable getGeneratedValues() {
		return this.generatedKeys;
	}
}
