package server.database.schema.app;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class PreCQEThirdExample extends MaintenanceDatabaseSchema {


	public final IllTableSchema ill;
	public final TreatTableSchema treat;
	public final EffectsTableSchema effects;
	
	public PreCQEThirdExample() throws DatabaseException {
		super("PreCQE Third Example");
		
		this.ill = new IllTableSchema();
		this.treat = new TreatTableSchema();
		this.effects = new EffectsTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.ill );
		this.appTables.add( this.treat );
		this.appTables.add( this.effects );
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillIll(db);
		this.fillTreat(db);
	}
	
	public void fillIll(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" );
		sql.set( IllTableSchema.DIAGNOSIS, "Cancer" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" );
		sql.set( IllTableSchema.DIAGNOSIS, "BrokenLeg" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Lisa" );
		sql.set( IllTableSchema.DIAGNOSIS, "Aids" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Lisa" );
		sql.set( IllTableSchema.DIAGNOSIS, "BrokenArm" );
		db.insert(this.ill);
	}
	
	public void fillTreat(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineB" );
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Lisa" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineA" );
		db.insert(this.treat);
	}
		
	public void fillEffects(SQLDatabase db) throws DatabaseException {	
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "LowWorkCapacity" );
		db.insert(this.effects);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "HighCosts" );
		db.insert(this.effects);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Lisa" );
		sql.set( TreatTableSchema.PRESCRIPTION, "LowWorkCapacity" );
		db.insert(this.effects);
	}

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		super.createDefaultAccessRights(db);
		int userId = super.createTestUser(db, 1, "PreCQE");
		try {
			this.fillPrior(db.getUserManagement().load(userId));
			this.fillConfidentialityPolicy(db.getUserManagement().load(userId));
		} catch (UnknownUserException e) {
			throw new DatabaseException("User does not exists, check your code.", e);
		}
	}
	
	public void fillPrior(User user) throws DatabaseException {
		try {
			PriorUser prior = user.getPriorUser();
			prior.add(new Formula("FORALL X (treat(X,\"MedicineA\") IMPL (ill(X,\"Aids\") OR ill(X,\"Cancer\")))"));
			prior.add(new Formula("FORALL X (treat(X,\"MedicineB\") IMPL (ill(X,\"Cancer\") OR ill(X,\"Flu\")))"));
			prior.add(new Formula("FORALL X (ill(X,\"BrokenLeg\") IMPL effects(X,\"HighCosts\"))"));
			prior.add(new Formula("FORALL X (ill(X,\"BrokenArm\") IMPL effects(X,\"LowWorkCapacity\"))"));
			prior.add(new Formula("FORALL X (ill(X,\"Cancer\") IMPL (effects(X,\"LowWorkCapacity\") AND effects(X,\"HighCosts\")))"));
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			policy.add(new Formula("EXISTS X ill(X,\"Cancer\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("EXISTS X ill(X, \"Aids\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add(new Formula("EXISTS X effects(X, \"HighCosts\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
}
