package server.database;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import exception.DatabaseException;

import server.data.Dictionary;
import server.data.PriorAll;
import server.data.SchemaConstraints;
import server.data.UserManagement;
import server.database.sql.OracleSQLDatabase;
import server.database.sql.SQLDatabase;

/**
 * Maintenance Database Abstraction.
 *
 * Each client is assigned with one MaintenanceDatabase instance,
 * but one MaintenanceDatabase instance can have multiple clients assigned.
 * 
 * Modifications of the maintenance database should only be done through this class.
 * Each data abstraction object implements the observer design pattern.
 * Use them if you wan't to be notified about changes.
 */
public class NewMaintenanceDatabase {
	// Cache for already loaded instances.
	private static final Map<DatabaseConfiguration,NewMaintenanceDatabase> dbCache = Collections.synchronizedMap( new HashMap<DatabaseConfiguration,NewMaintenanceDatabase>() );
	
	// Database connection.
	private SQLDatabase db;
	// Data stored in the maintenance database.
	private UserManagement userManagement;
	private Dictionary dictionary;
	private SchemaConstraints schemaConstraints;
	private PriorAll priorAll;

	/**
	 * Factory for NewMaintenanceDatabase instances.
	 * 
	 * A database connection is only established once for each
	 * different maintenance database. This is needed to do proper
	 * synchronization.
	 * 
	 * @param config
	 * @return
	 * @throws DatabaseException
	 */
	public static NewMaintenanceDatabase load( DatabaseConfiguration config ) throws DatabaseException {
		// Check cache.
		if ( NewMaintenanceDatabase.dbCache.containsKey(config) ) {
			return NewMaintenanceDatabase.dbCache.get(config);
		}
		
		// Create new database connection.
		NewMaintenanceDatabase db = new NewMaintenanceDatabase( new OracleSQLDatabase(config) );
		
		// Store in cache.
		NewMaintenanceDatabase.dbCache.put(config, db);
		
		return db;
	}

	/**
	 * Creates a new instance.
	 * Only to be called by {@link #load(DatabaseConfiguration)}.
	 * @param db Connection to the database.
	 * @throws DatabaseException 
	 */
	private NewMaintenanceDatabase( SQLDatabase db ) throws DatabaseException {
		this.db = db;
		this.dictionary = new Dictionary(db);
		this.schemaConstraints = new SchemaConstraints(db);
		this.priorAll = new PriorAll(db);
		this.userManagement = new UserManagement(db, priorAll, schemaConstraints);
	}

	/**
	 * Returns the database connection. Use this if you want to create
	 * your own SQL queries.
	 * Don't use this to modify data managed by the classes in the data package.
	 * @return
	 */
	public SQLDatabase getDb() {
		return this.db;
	}

	/**
	 * Returns the user management instance for this maintenance db.
	 * @return
	 */
	public UserManagement getUserManagement() {
		return this.userManagement;
	}

	/**
	 * Returns the dictionary instance for this maintenance db.
	 * @return
	 */
	public Dictionary getDictionary() {
		return this.dictionary;
	}

	/**
	 * Returns the schema constraints instance for this maintenance db.
	 * @return
	 */
	public SchemaConstraints getSchemaConstraints() {
		return this.schemaConstraints;
	}

	/**
	 * Returns the prior all instance for this maintenance db.
	 * @return
	 */
	public PriorAll getPriorAll() {
		return this.priorAll;
	}
}
