package server.data;

import java.util.LinkedList;
import java.util.List;

import exception.DatabaseException;

import server.database.schema.maintenance.DictionariesTableSchema;

/**
 * A DictionaryEntry is a list of possible values for a specific
 * attribute of a specific relation.
 * 
 * The get methods will only return immutable objects like Strings or
 * copies of mutable objects like Formulas.
 * This means any changes MUST be done through the set methods to
 * take any effect.
 */
public class DictionaryEntry {

	private Dictionary dictionary;
	private String identifier;
	private String dict_table;
	private String relation;
	private String attribute;
	private List<String> values;
	
	DictionaryEntry( Dictionary dictionary, String identifier, String dict_table, String relation, String attribute, List<String> values ) {
		this.identifier = identifier;
		this.dict_table = dict_table;
		this.relation = relation;
		this.attribute = attribute;
		this.values = values;
		this.dictionary = dictionary;
	}
	
	/**
	 * Returns the unique identifier of this relationname attribute pair.
	 * @return
	 */
	public String getIdentifier() {
		return this.identifier;
	}
	
	/**
	 * Returns the name of the table which contains the dictionary.
	 * @return
	 */
	public String getDictTable() {
		return this.dict_table;
	}
	
	/**
	 * Returns the relation's name.
	 * @return
	 */
	public String getRelation() {
		return this.relation;
	}
	
	/**
	 * Sets the relation's name to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param relation
	 * @throws DatabaseException 
	 */
	public void setRelation( String relation ) throws DatabaseException {
		this.relation = relation;
		this.dictionary.updateEntry( this, DictionariesTableSchema.RELATION, relation );
	}
	
	/**
	 * Returns the attribute's name.
	 * @return
	 */
	public String getAttribute() {
		return this.attribute;
	}
	
	/**
	 * Sets the attribute's name to a new value.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param attribute
	 * @throws DatabaseException 
	 */
	public void setAttribute( String attribute ) throws DatabaseException {
		this.attribute = attribute;
		this.dictionary.updateEntry( this, DictionariesTableSchema.ATTRIBUTE, attribute );
	}
	
	/**
	 * Returns a copy of all values for this attribute-relation pair.
	 * Modifying the copied list will have no effect on the dictionary.
	 * @return
	 */
	public List<String> getValues() {
		return new LinkedList<String>( this.values );
	}
	
	/**
	 * Sets the values for this attribute-relation pair to a new value.
	 * A copy of the given values will be created. Modifying the list afterwards will have no effect on the dictionary.
	 * 
	 * The modification will directly be reflected in the database.
	 * Observers will be notified.
	 * 
	 * @param values
	 */
/**	public void setValues( List<String> values ) {
		this.values = new LinkedList<String>( values );
		this.dictionary.updateEntry( this, ConfidentialityPolicyTableSchema.FORMULA, this.formula.toString() );
	}*/
}
