package server.censor.signature.experiment;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import exception.DatabaseException;
import exception.UnsupportedFormulaException;

import server.censor.signature.util.ConstantElement;
import server.censor.signature.util.MaintenanceUtil;
import server.censor.signature.util.SignatureConfidentialityPolicyItem;
import server.censor.signature.util.TemplateDependency;
import server.censor.signature.util.TemplateDependencyRow;
import server.database.AppDatabase;
import server.database.DatabaseConfiguration;
import server.database.NewMaintenanceDatabase;
import server.database.OracleSQLAppDatabase;
import server.database.cache.AppDatabaseSchemaCache;
import server.database.schema.Schema;
import server.database.schema.maintenance.SignaturesFlexiblePotSecAVTableSchema;
import server.database.schema.maintenance.SignaturesFlexiblePotSecTableSchema;
import server.database.schema.maintenance.SignaturesFlexibleSignaturesPAVSchema;
import server.database.schema.maintenance.SignaturesFlexibleSignaturesPSchema;
import server.database.schema.maintenance.SignaturesFlexibleSignaturesSchema;
import server.database.schema.maintenance.SignaturesUnflexiblePotSecSchema;
import server.database.schema.maintenance.SignaturesUnflexibleSignaturesSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.util.Tuple;


/**
 *  Diese Klasse dient der Vorbereitung des Experiments zum SignatureCensor.
 *  Das Experiment ist in "Signature-Based Inference-Usability Confinement for Relational 
 *  Databases under Functional and Join Dependencies" von Biskup etc. beschrieben.
 *  Details des Experimentaufbaus koennen der Datei 'Experimentbeschreibung_DBSEC12.txt' 
 *  entnommen werden.
 */
public abstract class ExperimentPreparation {
	
	/** Hilfsklasse fuer Maintenance-Aufgaben */
	protected MaintenanceUtil maintenanceUtil;
	
	/** Maintenance Datenbank */
	protected NewMaintenanceDatabase maintenanceDB;
	
	/** Applikations Datenbank */
	protected AppDatabase appDB;
	private AppDatabaseSchemaCache appDBcache;
	
	/** Verwendung des flexiblen oder unflexibler Ansatzes */
	protected boolean flexible;
	
	/** User fuer den das Experiment erstellt wird */
	protected int userID;
	
	/** Name der zu schuetzenden Relation */
	protected String relationname;
	
	
	/**
	 * Konstruktor.
	 * 
	 * @param maintenanceDB Maintenance-Datenbank
	 * @param appDB Applikations-Datenbank
	 * @param flexible true, wenn der flexible Ansatz verwendet werden soll, false fuer den unflexiblen Ansatz
	 * @param userID Benutzer fuer den das Experiment durchgefuehrt werden soll
	 * @throws DatabaseException
	 */
	public ExperimentPreparation(NewMaintenanceDatabase maintenanceDB, AppDatabase appDB, boolean flexible, int userID, String relationname) throws DatabaseException {
		this.maintenanceDB = maintenanceDB;
		this.appDB = appDB;
		this.flexible = flexible;
		this.userID = userID;
		this.relationname = relationname;
		this.maintenanceUtil = new MaintenanceUtil();
		this.appDBcache = new AppDatabaseSchemaCache(appDB.getSQLDatabase(), maintenanceDB);
	}
	
	/**
	 * Konstruktor.
	 * 
	 * @param dbConfig Datenbank-Konfiguration
	 * @param flexible true, wenn der flexible Ansatz verwendet werden soll, false fuer den unflexiblen Ansatz
	 * @param userID Benutzer fuer den das Experiment durchgefuehrt werden soll
	 * @throws DatabaseException
	 */
	public ExperimentPreparation(Tuple<DatabaseConfiguration,DatabaseConfiguration> dbConfig, boolean flexible, int userID, String relationname) throws DatabaseException {
		this.maintenanceDB = NewMaintenanceDatabase.load( dbConfig.getFirstElement() );
		this.appDB = new OracleSQLAppDatabase( dbConfig.getSecondElement() );
		this.flexible = flexible;
		this.userID = userID;
		this.relationname = relationname;
		this.maintenanceUtil = new MaintenanceUtil();
		this.appDBcache = new AppDatabaseSchemaCache(appDB.getSQLDatabase(), maintenanceDB);
	}
	
	/**
	 * Fuehrt die fuer das Experiment notwendigen Vorbereitungen durch.
	 * Dies beinhaltet das Zuruecksetzen der verwendeten Datenbank-Tabellen und
	 * die Generierung der potentiellen Geheimnisse, Signaturen und Instanzen.
	 * Es werden nur die Vorbereitungen fuer den ausgewaehlten Ansatz (unflexibel/
	 * flexibel) durchgefuehrt.
	 * 
	 * @param numberOfInstanceCopies	Anzahl der Kopien, die von der originalen Instanz 
	 * 									angefertigt werden sollen um die Performance auch 
	 * 									fuer groessere Instanzen messen zu koennen
	 */
	public void prepareExperiment(int numberOfInstanceCopies) throws UnsupportedFormulaException, DatabaseException {
		// Tabelle fuer partielle Instanzen erzeugen
		this.createPartialInstanceTable();
		
		// leere Datenbank
		this.clearDatabase();
		
		// Schreibe potentielle Geheimnisse in die Datenbank
		this.writePotSecIntoDatabase();
		
		// Schritt 1
		List<TemplateDependency> tds = generateTemplateDependencies();
		
		// Schritte 2+3
		this.generateTD_Psis(tds);
		
		// Schritt 4a
		this.generatePartialInstances(tds);
		
		// Schritt 4b
		List<TemplateDependency> instantiatedTDs = this.instantiateTDPsisWithPartialInstances(tds);
		
		// Schrit 5a inkl. dem Schreiben der instantiierten Template Dependencies in die Datenbank
		this.writePartialInstancesAndTemplateDependenciesIntoDatabase(instantiatedTDs);
		
		// Schritt 5b
		this.expandSetting(numberOfInstanceCopies);
		
		// Tabelle fuer partielle Instanzen loeschen
		this.deletePartialInstanceTable();
	}
	
	/**
	 * Diese Methode implementiert Schritt 1 des Experiment-Aufbaus:
	 * 
	 * Die verwendeten generischen Signaturen haengen von dem konkret 
	 * durchgefuehrten Experiment ab.
	 */
	protected abstract List<TemplateDependency> generateTemplateDependencies();
	
	/**
	 * Hilfsmethode, die das potentielle Geheimnis fuer die Template Dependency
	 * mit Index i zurueckliefert.
	 * Wie die potentiellen Geheimnisse aussehen, haengt vom konkret durchgefuehrten Experiment
	 * ab.
	 * 
	 * @param i	Index der Template Dependency, zu der das Geheimnis zurueckgeliefert werden soll
	 * @return	potentielles Geheimnis
	 */
	protected abstract SignatureConfidentialityPolicyItem getPotSec(int i) throws UnsupportedFormulaException, DatabaseException;
	
	/**
	 * Diese Methode implementiert die Schritte 2+3 des Experiment-Aufbaus:
	 * 
	 * Für jede generische Signatur  TD_i (i= 1,...,7)
	 * wird jeweils ein einziges potentielles Geheimnis Psi_i verwendet:
	 * 
	 * (exist X) R( X , Cancer_i , Smith_i )
	 * 
	 * wobei die Konstanten Cancer_i und  Smith_i für verschiedene i 
	 * paarweise verschieden sind. 
	 * 
	 * Damit wird jeweils die generischen Signatur TD_i in der Konklusion
	 * instantiiert, und dann wird diese Instantiierung in die Hypothesen
	 * weitergetragen.
	 * 
	 * Als Ergebnis liegt dann eine 
	 * teil-instantiierte Signatur TD_Psi_i vor.
	 */
	protected void generateTD_Psis(List<TemplateDependency> tds) throws DatabaseException, UnsupportedFormulaException {
		for( int i=0; i<7; ++i ) {
			this.instantiateTDWithPsi( tds.get(i), this.getPotSec(i) );
		}
	}
	
	/**
	 * Diese Methode implementiert Schritt 4a des Experiment-Aufbaus:
	 * http://www.google.de/
	 * Für jede so teil-instantiierte Signatur TD_Psi_i 
	 * wird jeweils eine Teilinstanz r_i wie folgt erzeugt:
	 * 
	 * -  behandle (ersetze) alle noch nicht instantiierten Symbole 
	 *    als (durch) paarweise verschiedene Konstanten, 
	 *    die auch verschieden von
	 *    Cancer_i und Smith_i sind
	 * 
	 * -  fasse das Tableau nun als Relation auf
	 * 
	 * -  "vervollständige" die so erhaltene Relation 
	 *     vermöge der beiden Join Dependencies durch Anwendung 
	 *     der chase procedure
	 * 
	 * @param tds mit den potentiellen Geheimnissen instantiierte Template Dependencies
	 * @throws DatabaseException
	 */
	protected void generatePartialInstances(List<TemplateDependency> tds) throws DatabaseException {
		// behandle noch nicht instantiierte Symbole als Konstanten
		for( int i=0; i<7; ++i ) {
			TemplateDependency td = new TemplateDependency(tds.get(i));
			for( int j=0; j<td.getPremiseCount(); ++j ) {
				// Symbole in den Praemissen durch Konstanten ersetzen
				TemplateDependencyRow premise = td.getPremise(j);
				for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
					if( premise.getElement(attribute).isSymbol() ) {
						// ersetze Symbol durch Konstante
						premise.setElement(attribute, new ConstantElement("td" + i + "_" + premise.getElement(attribute).getValue()));
					}
				}
			}
			// Symbole in der Konklusion durch Konstanten ersetzen
			TemplateDependencyRow conclusion = td.getConclusion();
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				if( conclusion.getElement(attribute).isSymbol() ) {
					// ersetze Symbol durch Konstante
					conclusion.setElement(attribute, new ConstantElement("td" + i + "_" + conclusion.getElement(attribute).getValue()));
				}
			}
			
			// bilde Teilrelation und vervollstaendige diese durch Anwendung der Join Dependencies
			SQLDatabase db = this.appDB.getSQLDatabase();
			String[] columnNames = db.getColumnNames(td.getRelationname());
			
			if( columnNames.length != td.getAttributeCount() ) {
				throw new DatabaseException("Error while writing partial instance: Number of attributes of the predicate doesn't match the number of columns in the database.", null);
			}
			
			// fuege durch Praemissen repraesentierte Tupel in die Datenbank ein
			for( int j=0; j<td.getPremiseCount(); ++j ) {
				TemplateDependencyRow premise = td.getPremise(j);
				SQLInsert sqlInsert = db.newInsert();
				for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
					sqlInsert.set( columnNames[attribute], premise.getElement(attribute).getValue() );
				}
				sqlInsert.set("partialInstance", i);
				sqlInsert.insert(relationname + "_PARTIAL");
			}
			// fuege durch Konklusion repraesentiertes Tupel in die Datenbank ein
			conclusion = td.getConclusion();
			SQLInsert sqlInsert = db.newInsert();
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				sqlInsert.set( columnNames[attribute], conclusion.getElement(attribute).getValue() );
			}
			sqlInsert.set("partialInstance", i);
			sqlInsert.insert(relationname + "_PARTIAL");
			
			// Vervollstaendigung durch Anwendung der Join Dependencies
			int insertedTupelCount;
			do {
				insertedTupelCount = 0;
				// Join Dependency [SD,SP]
				String union_querySD = "SELECT DISTINCT T1.PARTIALINSTANCE, T1.SYMPTOM, T1.DIAGNOSIS, T2.PATIENT FROM SIG_KRANKHEIT_PARTIAL T1, SIG_KRANKHEIT_PARTIAL T2 WHERE T1.PARTIALINSTANCE=T2.PARTIALINSTANCE AND T1.PARTIALINSTANCE=" + i + " AND T1.SYMPTOM=T2.SYMPTOM";
				String union_querySP = "SELECT DISTINCT T1.PARTIALINSTANCE, T1.SYMPTOM, T2.DIAGNOSIS, T1.PATIENT FROM SIG_KRANKHEIT_PARTIAL T1, SIG_KRANKHEIT_PARTIAL T2 WHERE T1.PARTIALINSTANCE=T2.PARTIALINSTANCE AND T1.PARTIALINSTANCE=" + i + " AND T1.SYMPTOM=T2.SYMPTOM";
				String minus_query = "SELECT DISTINCT * FROM SIG_KRANKHEIT_PARTIAL WHERE PARTIALINSTANCE=" + i;
				String query = "INSERT INTO " + relationname + "_PARTIAL (PARTIALINSTANCE, SYMPTOM, DIAGNOSIS, PATIENT) SELECT DISTINCT * FROM ( (" + union_querySD + " UNION " + union_querySP + ") MINUS ( " + minus_query + ") )";
				
				insertedTupelCount += db.dataManipulationQuery(query);
				// Join Dependency [DS,DP]
				String union_queryDS = "SELECT DISTINCT T1.PARTIALINSTANCE, T2.SYMPTOM, T1.DIAGNOSIS, T1.PATIENT FROM SIG_KRANKHEIT_PARTIAL T1, SIG_KRANKHEIT_PARTIAL T2 WHERE T1.PARTIALINSTANCE=T2.PARTIALINSTANCE AND T1.PARTIALINSTANCE=" + i + " AND T1.DIAGNOSIS=T2.DIAGNOSIS";
				String union_queryDP = "SELECT DISTINCT T1.PARTIALINSTANCE, T1.SYMPTOM, T1.DIAGNOSIS, T2.PATIENT FROM SIG_KRANKHEIT_PARTIAL T1, SIG_KRANKHEIT_PARTIAL T2 WHERE T1.PARTIALINSTANCE=T2.PARTIALINSTANCE AND T1.PARTIALINSTANCE=" + i + " AND T1.DIAGNOSIS=T2.DIAGNOSIS";
				query = "INSERT INTO " + relationname + "_PARTIAL (PARTIALINSTANCE, SYMPTOM, DIAGNOSIS, PATIENT) SELECT DISTINCT * FROM ( (" + union_queryDS + " UNION " + union_queryDP + ") MINUS ( " + minus_query + ") )";
				insertedTupelCount += db.dataManipulationQuery(query);
			} while( insertedTupelCount > 0 );
		}
	}
	
	/**
	 * Diese Methode implementiert Schritt 4b des Experiment-Aufbaus:
	 * 
	 * Instantiiere dann TD_Psi_i weiter mit r_i auf alle möglichen Weisen
	 * auf  m e h r f a c h  (hier nur doppelt) vorkommenden Symbolen
	 * (wobei zunächst ein Element von r_i auch für mehrere Hypothesen
	 * verwendet werden darf, obwohl das wahrscheinlich nur "Redundanz erzeugt").
	 * 
	 * 
	 * Wenn dann für solch eine weitere Instantiierung 
	 * eine Hypothese die Konlusion überdeckt, braucht diese weitere Instantierung 
	 * nicht mehr betrachtet zu werden.
	 * 
	 * @param tds mit den potentiellen Geheimnissen instantiierte Template Dependencies
	 * @throws DatabaseException
	 */
	protected List<TemplateDependency> instantiateTDPsisWithPartialInstances(List<TemplateDependency> tds) throws DatabaseException {
		// Instantiierung erfolgt auf Basis der partiellen Instanzen
		// -> Relationsnamen temporaer umbenennen
		for( TemplateDependency td : tds ) {
			td.setRelationname(relationname + "_PARTIAL");
		}
		
		List<Integer> attributeFilter = new LinkedList<Integer>();
		attributeFilter.add(1);
		attributeFilter.add(2);
		attributeFilter.add(3);
		List<TemplateDependency> instantiatedDependencies = TemplateDependency.generatePot_Sig(tds, this.appDB, attributeFilter);
		
		// Relationsnamen der resultierenden Dependencies wieder zuruecksetzen
		for( TemplateDependency td : instantiatedDependencies ) {
			td.setRelationname(relationname);
		}
		
		return instantiatedDependencies;
	}
	
	/**
	 * Diese Methode implementiert Schritt 5a des Experiment-Aufbaus:
	 * 
	 * Entferne redundante weitere Instantiierungen:
	 * wenn die Hypothesen einer solchen Instantiierung eine Obermenge der Hypothesen 
	 * einer anderen Instantiierung bilden, so braucht die "größere Instantiierung" 
	 * nicht mehr betrachtet zu werden.
	 * 
	 * 
	 * Zusaetzlich schreibt diese Methode die Instanz (zusammengesetzt aus den partiellen 
	 * Instanzen) und Template Dependencies in die Datenbank.
	 * 
	 * @param instantiatedTDs bereits instantiierte Template Dependencies
	 * @throws DatabaseException
	 */
	protected void writePartialInstancesAndTemplateDependenciesIntoDatabase(List<TemplateDependency> instantiatedTDs) throws DatabaseException {
		// Schreibe Instanz (zusammengesetzt aus den Teilinstanzen) in die Datenbank
		String query = "INSERT INTO " + relationname + " SELECT SYMPTOM, DIAGNOSIS, PATIENT FROM " + relationname + "_PARTIAL";
		this.appDB.getSQLDatabase().dataManipulationQuery(query);
		
		if( flexible ) {
			// Schreibe instanziierte Template Dependencies in die Datenbank
			for( TemplateDependency td : instantiatedTDs ) {
				this.maintenanceUtil.addSignatureFlexible(this.maintenanceDB.getDb(), appDBcache.getRelation(td.getRelationname()), td, this.userID);
			}
			
			// entferne Signaturen, die ein potentielles Geheimnis ueberdecken
			this.maintenanceUtil.removeSignaturesCoveringPotentialSecretsFlexible(this.maintenanceDB.getDb(), appDBcache.getRelation(relationname), relationname);
			
			// entferne Signaturen, die in einer anderen Signatur vollstaendig enthalten sind
			this.maintenanceUtil.removeSubsumedDependenciesFlexible(this.maintenanceDB.getDb(), appDBcache.getRelation(relationname), relationname);
			
			// entferne Luecken in den Indizes der Signaturen, die aufgrund des Loeschens von Signaturen entstanden sein koennen
			this.maintenanceUtil.removeSignatureIndexGapsFlexible(this.maintenanceDB.getDb());
		}
		else {
			// Schreibe instanziierte Template Dependencies in die Datenbank
			for( TemplateDependency td : instantiatedTDs ) {
				this.maintenanceUtil.addSignatureUnflexible(this.maintenanceDB.getDb(), appDBcache.getRelation(td.getRelationname()), td, this.userID);
			}
			
			// entferne Signaturen, die ein potentielles Geheimnis ueberdecken
			this.maintenanceUtil.removeSignaturesCoveringPotentialSecretsUnflexible(this.maintenanceDB.getDb(), appDBcache.getRelation(relationname), relationname);
			
			// entferne Signaturen, die in einer anderen Signatur vollstaendig enthalten sind
			this.maintenanceUtil.removeSubsumedDependenciesUnflexible(this.maintenanceDB.getDb(), appDBcache.getRelation(relationname), relationname);
			
			// entferne Luecken in den Indizes der Signaturen, die aufgrund des Loeschens von Signaturen entstanden sein koennen
			this.maintenanceUtil.removeSignatureIndexGapsUnflexible(this.maintenanceDB.getDb(), relationname);
		}
	}
	
	/**
	 * Diese Methode implementiert Schritt 5b des Experiment-Aufbaus:
	 * 
	 * Ausgehend von diesen Grundlagen,
	 * sollen nun Vollinstanzen wachsender Vervielfachung gebildet werden:
	 * 
	 * - für die "Vervielfachung" v = 1
	 *   bilde die Vollinstance vr1 durch Vereinigung der r_i, i= 1,..., 7
	 *   mitsamt der zugehörigen potentiellen Geheimnisse Psi_i, i= 1,..., 7
	 * 
	 * - für die "Vervielfachungen" v = 10 , 100 , 1000
	 *   erstelle jeweils v viele paarweise verschiedene Duplikate
	 *   des Ergebnisses für v=1
	 */
	protected void expandSetting(int numberOfCopies) throws DatabaseException {
		// Vervielfache Instanz
		this.expandInstance(numberOfCopies);
		
		if( flexible ) {
			// Vervielfache potentielle Geheimnisse
			this.expandPotSecFlexible(numberOfCopies);
			// Vervielfache Signaturen
			this.expandSignaturesFlexible(numberOfCopies);
		}
		else {
			// Vervielfache potentielle Geheimnisse
			this.expandPotSecUnflexible(numberOfCopies);
			// Vervielfache Signaturen
			this.expandSignaturesUnflexible(numberOfCopies);
		}
	}
	
	/**
	 * Expandiert die Instanz.
	 * 
	 * @param numberOfCopies Anzahl der Vervielfaeltigungen
	 * @throws DatabaseException
	 */
	protected void expandInstance(int numberOfCopies) throws DatabaseException {
		SQLDatabase db = this.appDB.getSQLDatabase();
		
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("*").from( relationname );
		SQLTable table = sqlSelect.get();
		SQLInsert sqlInsert = db.newInsert();
		for ( int i=2; i <= numberOfCopies; i++ ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					sqlInsert.set( columnName,  table.getRow(row).get(columnName) + "_v" + i);
				}
				sqlInsert.insert( relationname );
				sqlInsert = db.newInsert();
			}
		}
	}
	
	/**
	 * Expandiert die potentiellen Geheimnisse des unflexiblen Ansatzes.
	 * 
	 * @param numberOfCopies Anzahl der Vervielfaeltigungen
	 * @throws DatabaseException
	 */
	protected void expandPotSecUnflexible(int numberOfCopies) throws DatabaseException {
		SQLDatabase db = this.maintenanceDB.getDb();
		
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesUnflexiblePotSecSchema.ID.name + ") as MAX_ID").from(SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relationname);
		SQLTable table = sqlSelect.get();
		int nextID = Integer.parseInt(table.getFirstRow()[0]) + 1;
				
		sqlSelect = db.newSelect().select("*").from(SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relationname).order_by(SignaturesUnflexiblePotSecSchema.ID, false);
		table = sqlSelect.get();
		SQLInsert sqlInsert = db.newInsert();
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesUnflexiblePotSecSchema.ID.name) ) {
						sqlInsert.set( columnName, nextID++ );
					}
					else if( columnName.equals(SignaturesUnflexiblePotSecSchema.USER_ID.name) ) {
						sqlInsert.set( columnName, table.getRow(row).get(columnName) );
					}
					else {
						if( table.getRow(row).get(columnName).equals("*") )
							sqlInsert.set( columnName,  "*" );
						else
							sqlInsert.set( columnName,  table.getRow(row).get(columnName) + "_v" + i);
					}
				}
				sqlInsert.insert( SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relationname );
				sqlInsert = db.newInsert();
			}
		}
	}
	
	/**
	 * Expandiert die potentiellen Geheimnisse des flexiblen Ansatzes.
	 * 
	 * @param numberOfCopies Anzahl der Vervielfaeltigungen
	 * @throws DatabaseException
	 */
	protected void expandPotSecFlexible(int numberOfCopies) throws DatabaseException {
		SQLDatabase db = this.maintenanceDB.getDb();
		
		// PotSec
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesFlexiblePotSecTableSchema.POT_SEC_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesFlexiblePotSec);
		SQLTable table = sqlSelect.get();
		int nextID = Integer.parseInt(table.getFirstRow()[0]) + 1;
				
		sqlSelect = db.newSelect().select("*").from(Schema.maintenanceDatabase.signaturesFlexiblePotSec).order_by(SignaturesFlexiblePotSecTableSchema.POT_SEC_ID, false);
		table = sqlSelect.get();
		SQLInsert sqlInsert = db.newInsert();
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesFlexiblePotSecTableSchema.POT_SEC_ID.name) ) {
						sqlInsert.set( columnName, nextID++ );
					}
					else {
						sqlInsert.set( columnName,  table.getRow(row).get(columnName));
					}
				}
				sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexiblePotSec );
				sqlInsert = db.newInsert();
			}
		}
		
		// PotSecAV
		sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesFlexiblePotSecAVTableSchema.POT_SEC_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesFlexiblePotSecAV);
		table = sqlSelect.get();
		nextID = Integer.parseInt(table.getFirstRow()[0]); // kein +1, wird in der ersten Iteration der for-Schleife gemacht
				
		sqlSelect = db.newSelect().select("*").from(Schema.maintenanceDatabase.signaturesFlexiblePotSecAV).order_by(SignaturesFlexiblePotSecAVTableSchema.POT_SEC_ID, false);
		table = sqlSelect.get();
		sqlInsert = db.newInsert();
		int lastID = -1;
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesFlexiblePotSecTableSchema.POT_SEC_ID.name) ) {
						if( lastID != Integer.parseInt(table.getRow(row).get(columnName)) ) {
							lastID = Integer.parseInt( table.getRow(row).get(columnName) );
							++nextID;
						}
						sqlInsert.set( columnName, nextID );
					}
					else {
						if( table.getRow(row).get(columnName).equals("*") )
							sqlInsert.set( columnName,  "*" );
						else
							sqlInsert.set( columnName,  table.getRow(row).get(columnName) + "_v" + i);
					}
				}
				sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexiblePotSecAV );
				sqlInsert = db.newInsert();
			}
		}
	}
	
	/**
	 * Expandiert die Signaturen des unflexiblen Ansatzes.
	 * 
	 * @param numberOfCopies Anzahl der Vervielfaeltigungen
	 * @throws DatabaseException
	 */
	protected void expandSignaturesUnflexible(int numberOfCopies) throws DatabaseException {
		SQLDatabase db = this.maintenanceDB.getDb();
		
		// ermittle die bisher groesste ID
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesUnflexibleSignaturesSchema.SIG_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesUnflexibleSignaturesKrankheit);
		SQLTable result = sqlSelect.get();
		int nextID;
		String maxID = result.getRow(0).get("MAX_ID");
		if( maxID == null ) {
			// noch keine Signaturen vorhanden
			nextID = 1;
		}
		else {
			nextID = Integer.parseInt( maxID );	// kein +1, wird in der ersten Iteration der for-Schleife gemacht
		}
		
		// Sortierung ist notwendig, fuer die ID-Generierung
		sqlSelect = db.newSelect();
		sqlSelect.select("*").from(Schema.maintenanceDatabase.signaturesUnflexibleSignaturesKrankheit).order_by(SignaturesUnflexibleSignaturesSchema.SIG_ID, false);
		SQLTable table = sqlSelect.get();
		SQLInsert sqlInsert = db.newInsert();
		int lastID = -1;
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesUnflexibleSignaturesSchema.SIG_ID.name) ) {
						if( lastID != Integer.parseInt(table.getRow(row).get(columnName)) ) {
							lastID = Integer.parseInt( table.getRow(row).get(columnName) );
							++nextID;
						}
						sqlInsert.set( columnName, nextID );
					}
					else if( columnName.equals(SignaturesUnflexibleSignaturesSchema.USER_ID.name) ) {
						sqlInsert.set( columnName, table.getRow(row).get(columnName) );
					}
					else if( columnName.equals(SignaturesUnflexibleSignaturesSchema.FLAG_OLD.name) || columnName.equals(SignaturesUnflexibleSignaturesSchema.FLAG_IMPL.name) ) {
						sqlInsert.set( columnName, "0" );
					}
					else {
						if( ! table.getRow(row).get(columnName).equals("*") )
							sqlInsert.set( columnName,  table.getRow(row).get(columnName) + "_v" + i);
						else {
							sqlInsert.set( columnName,  "*");
						}
					}
				}
				sqlInsert.insert( SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relationname );
				sqlInsert = db.newInsert();
			}
		}
	}
	
	/**
	 * Expandiert die Signaturen des flexiblen Ansatzes.
	 * 
	 * @param numberOfCopies Anzahl der Vervielfaeltigungen
	 * @throws DatabaseException
	 */
	protected void expandSignaturesFlexible(int numberOfCopies) throws DatabaseException {
		SQLDatabase db = this.maintenanceDB.getDb();
		
		// Signatures
		// ermittle die bisher groesste ID
		SQLSelect sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesFlexibleSignaturesSchema.SIG_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesFlexibleSignatures);
		SQLTable result = sqlSelect.get();
		int nextID;
		String maxID = result.getRow(0).get("MAX_ID");
		if( maxID == null ) {
			// noch keine Signaturen vorhanden
			nextID = 1;
		}
		else {
			nextID = Integer.parseInt( maxID ) + 1;
		}
		
		// Sortierung ist notwendig, fuer die ID-Generierung
		sqlSelect = db.newSelect();
		sqlSelect.select("*").from(Schema.maintenanceDatabase.signaturesFlexibleSignatures).order_by(SignaturesFlexibleSignaturesSchema.SIG_ID, false);
		SQLTable table = sqlSelect.get();
		SQLInsert sqlInsert = db.newInsert();
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesFlexibleSignaturesSchema.SIG_ID.name) ) {
						sqlInsert.set( columnName, nextID++ );
					}
					else {
						sqlInsert.set( columnName, table.getRow(row).get(columnName) );
					}
				}
				sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexibleSignatures );
				sqlInsert = db.newInsert();
			}
		}
		
		// SignaturesP
		// ermittle die bisher groesste ID
		sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesFlexibleSignaturesPSchema.SIG_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP);
		result = sqlSelect.get();
		maxID = result.getRow(0).get("MAX_ID");
		if( maxID == null ) {
			// noch keine Signaturen vorhanden
			nextID = 1;
		}
		else {
			nextID = Integer.parseInt( maxID );	// kein +1, wird in der ersten Iteration der for-Schleife gemacht
		}
		
		// Sortierung ist notwendig, fuer die ID-Generierung
		sqlSelect = db.newSelect();
		sqlSelect.select("*").from(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP).order_by(SignaturesFlexibleSignaturesPSchema.SIG_ID, false);
		table = sqlSelect.get();
		sqlInsert = db.newInsert();
		int lastID = -1;
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesFlexibleSignaturesPSchema.SIG_ID.name) ) {
						if( lastID != Integer.parseInt(table.getRow(row).get(columnName)) ) {
							lastID = Integer.parseInt( table.getRow(row).get(columnName) );
							++nextID;
						}
						sqlInsert.set( columnName, nextID );
					}
					else {
						sqlInsert.set( columnName, table.getRow(row).get(columnName) );
					}
				}
				sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexibleSignaturesP );
				sqlInsert = db.newInsert();
			}
		}
		
		// SignaturesPAV
		// ermittle die bisher groesste ID
		sqlSelect = db.newSelect();
		sqlSelect.select("MAX(" + SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name + ") as MAX_ID").from(Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV);
		result = sqlSelect.get();
		maxID = result.getRow(0).get("MAX_ID");
		if( maxID == null ) {
			// noch keine Signaturen vorhanden
			nextID = 1;
		}
		else {
			nextID = Integer.parseInt( maxID );	// kein +1, wird in der ersten Iteration der for-Schleife gemacht
		}
		
		// Sortierung ist notwendig, fuer die ID-Generierung
		sqlSelect = db.newSelect();
		sqlSelect.select("*").from(Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV).order_by(SignaturesFlexibleSignaturesPAVSchema.SIG_ID, false);
		table = sqlSelect.get();
		sqlInsert = db.newInsert();
		lastID = -1;
		for ( int i=2; i <= numberOfCopies; ++i ) {
			for ( int row=0; row < table.getRowCount(); ++row ) {
				for ( String columnName : table.getColumnNames() ) {
					if( columnName.equals(SignaturesFlexibleSignaturesPAVSchema.SIG_ID.name) ) {
						if( lastID != Integer.parseInt(table.getRow(row).get(columnName)) ) {
							lastID = Integer.parseInt( table.getRow(row).get(columnName) );
							++nextID;
						}
						sqlInsert.set( columnName, nextID );
					}
					else {
						if( ! table.getRow(row).get(columnName).equals("*") )
							sqlInsert.set( columnName,  table.getRow(row).get(columnName) + "_v" + i);
						else {
							sqlInsert.set( columnName,  "*");
						}
					}
				}
				sqlInsert.insert( Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV );
				sqlInsert = db.newInsert();
			}
		}
	}
	
	/**
	 * Erzeugt eine temporaere Tabelle zum Aufbau der verwendeten Instanz.
	 */
	protected void createPartialInstanceTable() throws DatabaseException {
		String sql = "CREATE TABLE \"" + relationname + "_PARTIAL\" (\"PARTIALINSTANCE\" NUMBER, \"SYMPTOM\" VARCHAR2(400 BYTE), \"DIAGNOSIS\" VARCHAR2(400 BYTE), \"PATIENT\" VARCHAR2(400 BYTE))";
		this.appDB.getSQLDatabase().rawQuery(sql);
	}
	
	/**
	 * Loescht die temporaere Tabelle zum Aufbau der verwendeten Instanz.
	 */
	protected void deletePartialInstanceTable() throws DatabaseException {
		String sql = "DROP TABLE \"" + relationname + "_PARTIAL\"";
		this.appDB.getSQLDatabase().rawQuery(sql);
	}
	
	/**
	 * Setzt die fuer das Experiment verwendeten Relationen zurueck.
	 */
	protected void clearDatabase() throws DatabaseException {
		this.appDB.truncateRelation(relationname);
		
		SQLDatabase db = this.maintenanceDB.getDb();
		if( flexible ) {
			// Maintenance-Tabellen fuer den flexiblen Ansatz leeren
			db.truncateTable(Schema.maintenanceDatabase.signaturesFlexiblePotSec.name);
			db.truncateTable(Schema.maintenanceDatabase.signaturesFlexiblePotSecAV.name);
			db.truncateTable(Schema.maintenanceDatabase.signaturesFlexibleSignatures.name);
			db.truncateTable(Schema.maintenanceDatabase.signaturesFlexibleSignaturesP.name);
			db.truncateTable(Schema.maintenanceDatabase.signaturesFlexibleSignaturesPAV.name);
		}
		else {
			// Maintenance-Tabellen fuer den unflexiblen Ansatz leeren
			db.truncateTable(SignaturesUnflexiblePotSecSchema.TABLE_PREFIX + relationname);
			db.truncateTable(SignaturesUnflexibleSignaturesSchema.TABLE_PREFIX + relationname);
		}
	}
	
	/**
	 * Schreibt die potentiellen Geheimnisse in die Datenbank.
	 * 
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 */
	protected void writePotSecIntoDatabase() throws UnsupportedFormulaException, DatabaseException{
		// fuelle Tabelle fuer den unflexiblen Ansatz
		for( int i=0; i<7; ++i ) {
			SignatureConfidentialityPolicyItem item = this.getPotSec(i);
			String relationname = item.getRelationname().toUpperCase();
			
			if( flexible )
				this.maintenanceUtil.addConfidentialityPolicyItemFlexible( this.maintenanceDB.getDb(), this.appDBcache.getRelation(relationname), item );
			else
				this.maintenanceUtil.addConfidentialityPolicyItemUnflexible( this.maintenanceDB.getDb(), this.appDBcache.getRelation(relationname), item );
		}
	}
	
	/**
	 * Hilfsmethode, die die angegebene Template Dependency mit dem angegebenen
	 * potentiellem Geheimnis instantiiert. Die Template Dependency wird dabei
	 * direkt modifiziert.
	 * 
	 * @param td	Template Dependency, die mit dem pot. Geheimnis instantiiert wird
	 * @param psi	potentielles Geheimnis, das fuer die Instantiierung verwendet wird
	 */
	protected void instantiateTDWithPsi(TemplateDependency td, SignatureConfidentialityPolicyItem psi) throws UnsupportedFormulaException {
		Set<Integer> attributesWithConstants = psi.getAttributesWithConstants();
		
		// pruefe ob alle Attribute, die in psi mit Konstanten belegt sind in td eine ausgezeichnete Variable besitzen
		if( td.getAttributesContainingDistinguishedSymbols().containsAll(attributesWithConstants) ) {
			// instantiiere die Template Dependency mit phi
			
			for( int attribute : attributesWithConstants ) {
				String constantValue = psi.getAttributeValue(attribute);
				String distinguishedSymbol = td.getConclusion().getElement(attribute).getValue();
				
				// instantiiere Konklusion
				td.getConclusion().setElement(attribute, new ConstantElement(constantValue));
				
				// propagiere Instantiierung in die Praemissen
				for( int i=0; i<td.getPremiseCount(); ++i ) {
					TemplateDependencyRow premise = td.getPremise(i);
					if( premise.getElement(attribute).isDistinguishedSymbol() && premise.getElement(attribute).getValue().equals(distinguishedSymbol) ) {
						premise.setElement(attribute, new ConstantElement(constantValue) );
					}
				}
			}
		}
		else {
			throw new UnsupportedFormulaException("Fehler bei der Instanziierung einer Template Dependency mit einem potentiellem Geheimnis: TD enthaelt nicht fuer alle Konstanten im pot. Geheimnis eine ausgezeichnete Variable.");
		}
	}	
}
