package server.censor;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

import server.censor.Censor;
import server.core.Server;

public class CensorManagement {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.censor");
	
	private static String[] censors = {
		"server.censor.basic.AccessControlCensor",
		"server.censor.basic.DirectAccessCensor",
		"server.censor.combined.CombinedCensor",
		"server.censor.combined.OpenCombinedCensor",
		"server.censor.lying.LyingCensor",
		"server.censor.lying.OpenLyingCensor",
		//"server.censor.lying.LyingUpdateCensor",
		"server.censor.precqe.PreCQECensor",
		"server.censor.refusal.RefusalCensor",
		"server.censor.refusal.RefusalUpdateCensor",
		"server.censor.signature.FlexibleStaticSignatureCensor",
		"server.censor.signature.UnflexibleStaticSignatureCensor"
	};
	
	private Server server;
	private Map<String,Censor> loadedCensors;
	
	/**
	 * Creates a new CensorManangement instance.
	 * All available censors will be loaded.
	 * 
	 * The NotificationHub must be initialized before constructing the
	 * CensorManagement!
	 * 
	 * @param server
	 */
	public CensorManagement(Server server) {
		this.server = server;
		this.loadedCensors = Collections.synchronizedMap( new HashMap<String,Censor>() );

		for ( String censor : censors ) {
			logger.info("Loading censor: "+censor);
			
			try {
				this.loadCensor(censor);
			} catch (Exception e) {
				logger.error("Unable to load censor: "+censor, e);
			}
		}
	}
	
	/**
	 * Loads a censor and creates one instance.
	 * The censor is identified by the full package path.
	 * 
	 * @param path The full package name of the censor (e.g. "server.censor.LyingCensor").
	 * @return The loaded server instance. Won't be null (exceptions will be thrown if the censor couldn't be found).
	 * @throws IllegalArgumentException
	 * @throws SecurityException
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws NoSuchMethodException
	 * @throws ClassNotFoundException
	 */
	public Censor loadCensor( String path ) throws IllegalArgumentException, SecurityException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException {
		Censor censor = (Censor)Censor.class.getClassLoader().loadClass(path).getConstructor(Server.class).newInstance(this.server);
		this.loadedCensors.put(censor.getName(), censor);
		return censor;
	}
	
	/**
	 * Returns a previously loaded censor instance which is identified by its name.
	 * @param name The unique name of the censor.
	 * @return Censor instance that was loaded at startup. null if no censor with the name was loaded.
	 */
	public Censor getCensor( String name ) {
		logger.debug("get censor: "+name);
		logger.debug(this.loadedCensors);
		return this.loadedCensors.get(name);
	}
}
