package client.gui.usermanagement;

import java.io.IOException;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import notification.serverToClient.ExceptionNotification;

import client.core.Client;

import user.Role;
import user.User;

/**
 * GUI-Model: Fuellt eine Tabelle mit allen verfuegbaren Benutzern. Hat die
 * Spalten Name und Rolle.
 * @author schalge
 */
public class UserModel extends AbstractTableModel {
	private static final long serialVersionUID = 6039964524777969334L;

	private List<User> users;
	
	public UserModel( List<User> users ) {
		this.users = users;
	}
	
	/**
     * Gibt die Anzahl der Zeilen (Benutzer) zurueck.
     * @return Anzahl der Zeilen in der Liste.
     */
	@Override
    public int getRowCount() {
        return users.size();
    }

    /**
     * Gibt die Anzahl der Spalten zurueck. Hier immer 2.
     * @return Die Zahl 2.
     */
    @Override
    public int getColumnCount() {
        return 4;
    }

    /**
     * Gibt den String-Wert zum fuellen der Tabelle zurueck.
     * @param rowIndex Zeilennummer
     * @param columnIndex Spaltennummer.
     * @return String, der in dieser Zelle stehen soll.
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
    	if ( columnIndex == 0 )
    		return this.users.get(rowIndex).getUsername();
    	else if ( columnIndex == 1 ) {
    		if ( this.users.get(rowIndex).getPassword() == null )
    			return "<hidden>";
    		else
    			return this.users.get(rowIndex).getPassword();
    	}
    	else if ( columnIndex == 2 )
    		return this.users.get(rowIndex).getRole().getName();
    	else
    		return this.users.get(rowIndex).getCensor();
    }
    
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		User user = this.users.get( rowIndex );
		try {
			if ( columnIndex == 0 ) {
				String username = (String)aValue;
				if ( username.length() > 0 ) {
					if ( Client.GUI.editUser(user.getUserId(), username, null, null, null) ) {
						user.setUsername( username );
					}
				}
			} else if ( columnIndex == 1 ) {
				String password = (String)aValue;
				if ( password.length() > 0 ) {
					if ( Client.GUI.editUser(user.getUserId(), null, password, null, null) ) {
						user.setPassword( password );
					}
				}
			} else if ( columnIndex == 2 ) {
				Role role = (Role)aValue;
				if ( Client.GUI.editUser(user.getUserId(), null, null, role, null) ) {
					user.setRole( role );
				}
			} else {
				String censor = (String)aValue;
				if ( Client.GUI.editUser(user.getUserId(), null, null, null, censor) ) {
					user.setCensor( censor );
				}
			}
			
			this.fireTableRowsUpdated(rowIndex, rowIndex);
		} catch ( IOException e ) {
			Client.GUI.showError("Failed to edit user.", e, ExceptionNotification.NONCRITICAL);
		} catch ( NumberFormatException e ) {
			Client.GUI.showError("Only numbers are allowed as state id.", e, ExceptionNotification.NONCRITICAL);
		}
	}

    /**
     * Gibt die Spaltennamen fuer die Ueberschrift der Tabelle zurueck.
     * @param column Spaltennummer
     * @return Spalte 1: Name, Spalte2: Rolle
     */
    @Override
    public String getColumnName(int column) {
        if ( column == 0 ) {
            return "Name";
        } else if ( column == 1 ) {
        	return "Password";
        } else if ( column == 2 ) {
            return "Role";
        } else if ( column == 3 ) {
            return "Censor";
        } else {
            return "Column "+column;
        }
    }
    
    /**
     * Gibt den Benutzer in der angeforderten Zeile zurueck.
     * @param index Index der Zeile.
     * @return Ein Benutzer.
     */
    public User getUser( int index ) {
    	return this.users.get(index);
    }
    
    /**
     * Gibt alle in der Tabelle enthaltenen Benutzer zurueck.
     * @return Liste aller Benutzer.
     */
    public List<User> getUsers() {
    	return this.users;
    }
    
    /**
     * Fuegt einen neuen Benutzer am Ende der Liste ein.
     * @param user
     */
    public void addUser( User user ) {
    	this.users.add( user );
    	this.fireTableRowsInserted( getRowCount()-1, getRowCount()-1 );
    }
    
    /**
     * Loescht einen bestimmten Benutzer aus der Liste.
     * @param user
     */
    public void deleteUser( User user ) {
    	int index = this.users.indexOf( user );
    	if ( index >= 0 ) {
    		this.deleteUser(index);
    	}
    }
    
    /**
     * Loescht den Benutzer an einer bestimmten Stelle in der Liste.
     * @param index
     */
    public void deleteUser( int index ) {
    	this.users.remove( index );
    	this.fireTableRowsDeleted( index, index );
    }
    
    /**
     * Direktes Manipulieren des Benutzernamen erlauben.
     */
    @Override
    public boolean isCellEditable(int row, int col) {
    	return true;
    }
}
