package de.dwds.ddc.client;

import org.json.*;

/** class for query responses from ddc server, used by DDCClient */
public class DDCQueryResponse extends DDCJsonObject {
	//-----------------------------------------------------------------
	// properties (all public, because private data is annoying)
	
	/** server "internal status" flag (0:success) */
	public int istatus_;
	
	/** server "network status" flag (0:success) */
	public int nstatus_;
	
	/** server error message, if any; empty or null for none */
	public String error_;
	
	/** total number of hits found */
	public int   nhits_;
	
	/** distribution of hits over subcorpora */
	public String dhits_;
	
	/** total number of matching documents containing hits (only if server option ShowNumberOfRelevantDocuments is set) */
	public int   ndocs_;
	
	/** distribution of document-hits over subcorpus documents (only if server option ShowNumberOfRelevantDocuments is set) */
	public String ddocs_;
	
	/** logical index (1-based) of last hit on the current "page" */
	public int   end_;
	
	/** underlying hit data */
	public DDCHit[] hits_;
	
	//-----------------------------------------------------------------
	// constructors etc
	
	/** default constructor */
	DDCQueryResponse() {
		super();
		istatus_ = 0;
		nstatus_ = 0;
		error_ = null;
		nhits_ = 0;
		dhits_ = null;
		ndocs_ = 0;
		ddocs_ = null;
		end_   = 0;
		hits_  = null;
	}
	
	/** constructor from json string */
	DDCQueryResponse(String jsonStr) throws JSONException
	{ super(jsonStr); }

	/** constructor from json object*/
	DDCQueryResponse(JSONObject jsonObj) throws JSONException
	{ super(jsonObj); }

	//-----------------------------------------------------------------
	// parsing

	/** populate object by parsing ddc-2.x json response object */
	public void fromJson(JSONObject jsonObj) throws JSONException
	{
		super.fromJson(jsonObj);
		istatus_ = obj.optInt("istatus_",-1);
		nstatus_ = obj.optInt("nstatus_",-1);
		error_ = obj.optString("error_");
		nhits_ = obj.optInt("nhits_");
		dhits_ = obj.optString("dhits_");
		ndocs_ = obj.optInt("ndocs_");
		ddocs_ = obj.optString("ddocs_");
		end_   = obj.optInt("end_");

		//-- parse hits
		JSONArray jhits = obj.getJSONArray("hits_");
		hits_ = new DDCHit[jhits.length()];
		for (int i=0; i < hits_.length; i++) {
			hits_[i] = new DDCHit(jhits.getJSONObject(i));
		}
	}
	
	//-----------------------------------------------------------------
	// basic API
	
	/** @return true iff response represents a successful query result */
	public boolean isSuccess() {
		return istatus_==0 && nstatus_==0 && (error_==null || error_.length()==0);
	}
			
	/** @return error message, or null if no error */
	public String getError()
	{ return error_; }

	/** @return total number of hits found */
	public int getNHits()
	{ return nhits_; }
	
	/** @return number of documents containing hits (if server flag ShowNumberOfRelevantDocuments is set) */
	public int getNDocs()
	{ return ndocs_; }
	
	/** @return logical index of last returned hit (1-based) */
	public int getEnd()
	{ return end_; }
	
	/** @return an array of DDCHit objects representing the current page of hits */ 
	public DDCHit[] getHits()
	{ return hits_; }
}
