function [y1] = BQ05_RB50TS(x1)
% BQ05_RB50TS is neural network simulation function.
% It predicts TI BQ25505 efficiency according to the input voltage and
% current in addition to the storage voltage.
% 
% [y] = BQ05_RB50TS(x) takes these arguments as:
%   x = [Vin, Vstr, Ii]'
%       Voltage values in V (Volt)
%       Current value in uA (micro Ampere)
% and returns:
%   y = efficiency (in percent)
% 
% This model has a single hidden layer including 50 neurons with radial basis transfer function 
% and  a single neuron tangent sigmoid transfer function output layer.

% Coperight: Mojtaba Masoudinejad, 2018, Dortmund
% mojtaba.masoudinejad@tu-dortmund.de

% ===== NEURAL NETWORK CONSTANTS =====

% Input 1
x1_step1.xoffset = [0.328;1.5999;0.5];
x1_step1.gain = [0.447727781508843;0.769201184569824;1.99005965203807e-05];
x1_step1.ymin = -1;

% Layer 1
b1 = [-130.77510023115254967;2.8295615378098899484;-153.38126647545993819;-66.446585245089025307;33.739009010584624093;-4.1574300585949108822;-243.58975797330859336;124.77682425339305894;56.64146349461638863;-14.390070665894555191;-669.73608706964046178;251.9931029949867991;-79.080596945547341647;145.64447632390371723;-1.1953887526992117429;-551.63665457122499447;-658.50997528896903077;53.513916515987553169;5.8455517641135275042;19.214805701832091245;-117.68767349701919045;-435.80706191413662509;-34.455901529781726822;387.57915719822744904;-1.0292668870215575527;427.77924297678157473;-5.3115437189559759901;42.696250735002855947;20.105879272525712054;-22.952182234912246628;-23.111863876941431073;49.448564373259259241;17.156074242906417027;-40.643806607969011679;91.068487452895368506;-56.433873126345389437;29.63191151615967911;0.77665308350002204563;37.039150751145164975;58.904101193534010861;839.55805008635297781;37.334532041073863695;29.957190646918039789;2.720232852571537574;-17.516133007769006724;26.493237302091660723;-25.527374163900109494;-1350.5635751519871519;25.934520036506512497;-49.791760075175382383];
IW1_1 = [1.4093295534798189195 27.979664104022898385 -102.48821708345779768;-10.545746586887581486 5.9325853207057832606 0.28481953432267637094;84.108353894918252536 -34.694193316902335766 -160.09947333226446631;-0.62851277600815735891 -27.729224096282575118 -47.92997809103702167;-22.185280978205188518 16.435035266982229274 36.70260723891517074;10.476445806767005209 -5.7746186495244264947 -0.58089794569610408992;23.726191504622555328 -14.016002368783526677 -248.04734013937090253;-22.625393099443673606 -37.231942735363425356 91.17836924962355738;-49.321856718881200266 -48.374503483067748277 14.424008321487074014;-0.68153669407851413808 -1.0185873550877257987 -10.825948570603546983;-0.96545628296038965566 0.61575105385134221159 -666.12820614712597944;-516.65825396796390123 -167.24257883539593195 -33.901342604020484828;64.161938827743682623 -26.640934343468718026 -83.994762737573324785;-1.5358192676459099246 -29.168878392092473462 116.16182417946717464;0.57097849516337473297 -0.78552176179277788393 0.45982887227938618047;-2.4895248994730545533 -0.048047924541252645669 -547.53417885592091352;-3.1008698013689000028 -0.063270104087153444805 -654.25456383270739025;5.2483150294653269086 13.962262723287645727 34.432764976653423616;-35.251188428998524671 20.97989798739067524 0.65552774280821934916;-8.4376134232297630433 6.2796706629690062229 19.42272998878099699;415.3200308669680112 1.2928825859711210455 -77.246188657531661192;-0.42460452248797103669 -0.14180158784779467451 -434.55169737419657849;64.780378449317197465 -11.898054371691864972 -63.941040020453492332;-54.370358489433165516 31.358492612000055999 395.05517442266307171;-0.079222064704343120312 0.20765471625369558151 -1.1549428721975782963;0.41641131660746905085 0.13577091408206531931 426.49974801059698848;-92.193089650299668847 39.848158177808485902 -4.6675733314293577081;-79.181915838773548444 339.29604519584137279 -74.571297330858627106;-7.0476563223077715037 5.0858894827279534923 20.634977453535672254;-29.990985232978676578 82.606978226224924811 -69.737070981882737897;-30.200844913139775372 83.164649063601245871 -70.20043830662520179;77.834082078767437451 41.446914648152805682 43.373419594554256662;-7.9056041636560046015 5.8997825904420029275 17.730235334267614888;75.546746883004757933 5.091314503640436584 -99.063323484658170059;125.41678740406771908 -88.262694816392723851 116.68647220369136619;-88.668638231921107717 -11.917374445382671766 -29.652107688833094556;-53.566122137697767869 -3.8383987637299621376 74.547362570047980057;1.6614513915544402156 -0.39189361259259519743 0.10198974818005107634;82.932012896284305725 -72.601840711510760684 6.4935974043102344666;18.936302246363347024 -80.817944511450434675 32.518335548930338064;1.2649049525889710655 -0.51831532840813632212 836.07210676205693289;83.606509384639224436 -73.176508764240907112 6.56179083569856747;-54.162678358139139334 -3.8627845167717458352 75.363858009527277204;0.64777938022401571061 -0.014604933548669920759 -0.023673933751646284579;8.0816053964164318302 -6.0288413176795039305 -18.179420507027320042;21.170440983379197775 23.699893112102248693 -0.63614873860763865299;-20.270346957474934868 -23.009017718510911266 0.55106123371674464728;-0.070595076443013354828 0.15519445159529890454 -1347.8593700574576815;20.65807539560960393 23.287422981791220167 -0.59060853733379370389;-49.029619020705261789 -2.1985564560994532712 -22.411501347394956696];

% Layer 2
b2 = 1.1887106358569548004;
LW2_1 = [-19.595793233744174699 -0.60909500028186702725 0.19149179703115523643 0.084772189780354861854 -0.12575437381902598855 -0.46497627780692968313 0.045398805714767460906 -0.032782041405754240582 -0.026282506661234605883 -14.947322235830897696 440.37554657016767123 0.045914129742909752518 -0.23322727459274322781 18.376300778711350148 -0.37829225501837798751 234.08191844445386209 -58.414381625336204706 21.911939355171327293 -0.3565756299633374149 8.3976717839346068928 0.039748357279456955626 125.55850902080453579 0.050174949559214240824 -0.053059636350019825923 0.36595968104861492654 -134.71605391937598029 -0.047348321045274886776 0.042552711869885462281 -1.3637567865628861252 -5.8164469871879163065 5.8087698894338899436 0.0057604454730317286928 -46.500561262542575491 0.028581038975082018183 0.081459840530003393977 -0.045444054343810533214 -5.0094656760427200837 -0.20843819734912061303 4.8303788085021963639 0.014594976625067057965 -160.13428448235057999 -4.8195286001232204853 5.0308445186182026987 -65.178407783249099339 40.991722621472405308 -13.10657593055555914 -13.721679259747258328 -662.57589599858397378 26.735105963099268678 -0.044488709503979137483];

% Output 1
y1_step1.ymin = -1;
y1_step1.gain = 0.0200002226111087;
y1_step1.xoffset = 2.37595156899639e-05;

% ===== SIMULATION ========

% Dimensions
Q = size(x1,2); % samples

% Input 1
xp1 = mapminmax_apply(x1,x1_step1);

% Layer 1
a1 = radbas_apply(repmat(b1,1,Q) + IW1_1*xp1);

% Layer 2
a2 = tansig_apply(repmat(b2,1,Q) + LW2_1*a1);

% Output 1
y1 = mapminmax_reverse(a2,y1_step1);
end

% ===== MODULE FUNCTIONS ========

% Map Minimum and Maximum Input Processing Function
function y = mapminmax_apply(x,settings)
  y = bsxfun(@minus,x,settings.xoffset);
  y = bsxfun(@times,y,settings.gain);
  y = bsxfun(@plus,y,settings.ymin);
end

% Radial Basis Transfer Function
function a = radbas_apply(n,~)
  a = exp(-(n.*n));
end

% Sigmoid Symmetric Transfer Function
function a = tansig_apply(n,~)
  a = 2 ./ (1 + exp(-2*n)) - 1;
end

% Map Minimum and Maximum Output Reverse-Processing Function
function x = mapminmax_reverse(y,settings)
  x = bsxfun(@minus,y,settings.ymin);
  x = bsxfun(@rdivide,x,settings.gain);
  x = bsxfun(@plus,x,settings.xoffset);
end
