package server.start;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.plaf.basic.BasicTextFieldUI;
import javax.swing.text.JTextComponent;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import server.parser.Formula;

import exception.ParserException;
import exception.UnsupportedFormulaException;

/**
 * Zeigt eine graphische Baumstruktur fuer Formeln an.
 */
public class SyntaxTreeViewer extends JFrame {
	private static final long serialVersionUID = 2963063594220509082L;
	
	private JTree tree = null;
	private JTextField formulaInput = null;

	public SyntaxTreeViewer() {
        super();

        // Fensterzeug.
        this.setTitle( "Syntax Tree Viewer" );
        this.setPreferredSize( new Dimension(1000,800) );
        this.setDefaultCloseOperation( JFrame.EXIT_ON_CLOSE );
        this.getContentPane().setLayout( new BoxLayout(this.getContentPane(), BoxLayout.Y_AXIS) );

        this.formulaInput = new JTextField();
        this.formulaInput.setUI(new HintTextFieldUI("Input Formula here", true));
        this.formulaInput.setPreferredSize(new Dimension(50,20));
        this.formulaInput.setMaximumSize(new Dimension(Short.MAX_VALUE,20));
        this.formulaInput.addKeyListener(new KeyAdapter() {
        	@Override
        	public void keyTyped(KeyEvent e) {
        		if ( e.getKeyChar() == '\n' )
        		{
        			try {
        				SyntaxTreeViewer.this.updateTree(new Formula(SyntaxTreeViewer.this.formulaInput.getText()));
        			} catch (ParserException e1) {
        				// TODO Auto-generated catch block
        				e1.printStackTrace();
        			}
        			e.consume(); // Stop the event from propagating.
        		}
        	}
        });
        
        this.tree = new JTree();        
        this.tree.setModel( new DefaultTreeModel(null) );
        
        // Scrollbar hinzufuegen.
        JScrollPane treeView = new JScrollPane( tree );
        
        this.add( formulaInput );
        this.add(Box.createRigidArea(new Dimension(5,5)));
        this.add( treeView );

        this.pack();
        treeView.requestFocus();
        this.setVisible(true);
	}
	
	private void updateTree( Formula formula ) {
		// Baum anlegen.
        DefaultMutableTreeNode root = new DefaultMutableTreeNode( formula.toString() );
        formula.createTreeDump(root);
        
		this.tree.setModel( new DefaultTreeModel(root) );
		// Alle Nodes des Baums sichtbar machen.
        for ( int i = 0; i < tree.getRowCount(); i++ ) {
            this.tree.expandRow(i);
        }
	}
	
	public static void main(String[] args) throws ParserException, UnsupportedFormulaException {
		new SyntaxTreeViewer();
	}
	
	
	/**
	 * Class that provides a hint in the background of a textfield.
	 * Taken from http://stackoverflow.com/questions/1738966/java-jtextfield-with-input-hint
	 */
	class HintTextFieldUI extends BasicTextFieldUI implements FocusListener {

	    private String hint;
	    private boolean hideOnFocus;
	    private Color color;

	    public Color getColor() {
	        return color;
	    }

	    public void setColor(Color color) {
	        this.color = color;
	        repaint();
	    }

	    private void repaint() {
	        if(getComponent() != null) {
	            getComponent().repaint();           
	        }
	    }

	    public boolean isHideOnFocus() {
	        return hideOnFocus;
	    }

	    public void setHideOnFocus(boolean hideOnFocus) {
	        this.hideOnFocus = hideOnFocus;
	        repaint();
	    }

	    public String getHint() {
	        return hint;
	    }

	    public void setHint(String hint) {
	        this.hint = hint;
	        repaint();
	    }
	    public HintTextFieldUI(String hint) {
	        this(hint,false);
	    }

	    public HintTextFieldUI(String hint, boolean hideOnFocus) {
	        this(hint,hideOnFocus, null);
	    }

	    public HintTextFieldUI(String hint, boolean hideOnFocus, Color color) {
	        this.hint = hint;
	        this.hideOnFocus = hideOnFocus;
	        this.color = color;
	    }

	    @Override
	    protected void paintSafely(Graphics g) {
	        super.paintSafely(g);
	        JTextComponent comp = getComponent();
	        if(hint!=null && comp.getText().length() == 0 && (!(hideOnFocus && comp.hasFocus()))){
	            if(color != null) {
	                g.setColor(color);
	            } else {
	                g.setColor(comp.getForeground().brighter().brighter().brighter());              
	            }
	            int padding = (comp.getHeight() - comp.getFont().getSize())/2;
	            g.drawString(hint, 2, comp.getHeight()-padding-1);          
	        }
	    }

	    @Override
	    public void focusGained(FocusEvent e) {
	        if(hideOnFocus) repaint();

	    }

	    @Override
	    public void focusLost(FocusEvent e) {
	        if(hideOnFocus) repaint();
	    }
	    @Override
	    protected void installListeners() {
	        super.installListeners();
	        getComponent().addFocusListener(this);
	    }
	    @Override
	    protected void uninstallListeners() {
	        super.uninstallListeners();
	        getComponent().removeFocusListener(this);
	    }
	}
}
