package server.start;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Vector;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.UIManager;

import exception.DatabaseConfigurationException;
import exception.DatabaseException;

import server.database.DatabaseConfiguration;
import server.database.DatabaseConfigurationList;
import server.database.NewMaintenanceDatabase;
import server.database.schema.DatabaseSchema;
import server.database.schema.Schema;
import server.database.sql.OracleSQLDatabase;
import server.util.Tuple;

public class SchemaAdmin extends JFrame implements ActionListener {
	private static final long serialVersionUID = -7176045593879379860L;
	
	private JComboBox dbSelection;
	private JComboBox schemaSelection;
	private JButton createButton;
	private JButton dropButton;
	private JButton dropAllButton;
	private JButton fillButton;

	public SchemaAdmin() throws DatabaseConfigurationException {
		super();
		
		// try to set a nice theme
        try {
            UIManager.setLookAndFeel("com.sun.java.swing.plaf.nimbus.NimbusLookAndFeel");
        } catch ( Exception e) { }
		
		this.setTitle( "jCIE Schema Admin" );
		
		// exit application on window close
        this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        // Use Y Layout
        this.setLayout( new BoxLayout(this.getContentPane(), BoxLayout.Y_AXIS) );
        
        this.dbSelection = new JComboBox( this.getAvailableDatabases() );
        this.schemaSelection = new JComboBox( Schema.databases );
        
        this.createButton = new JButton("Create Schema");
        this.createButton.setAlignmentX(JButton.CENTER_ALIGNMENT);
        this.createButton.setMaximumSize(new Dimension(Short.MAX_VALUE,Short.MAX_VALUE));
        this.createButton.addActionListener( this );

        this.dropButton = new JButton("Drop Schema");
        this.dropButton.setAlignmentX(JButton.CENTER_ALIGNMENT);
        this.dropButton.setMaximumSize(new Dimension(Short.MAX_VALUE,Short.MAX_VALUE));
        this.dropButton.addActionListener( this );
        
        this.dropAllButton = new JButton("Drop all tables in app database");
        this.dropAllButton.setAlignmentX(JButton.CENTER_ALIGNMENT);
        this.dropAllButton.setMaximumSize(new Dimension(Short.MAX_VALUE,Short.MAX_VALUE));
        this.dropAllButton.addActionListener( this );
        
        this.fillButton = new JButton("Fill with contents");
        this.fillButton.setAlignmentX(JButton.CENTER_ALIGNMENT);
        this.fillButton.setMaximumSize(new Dimension(Short.MAX_VALUE,Short.MAX_VALUE));
        this.fillButton.addActionListener( this );
        
        this.add( dbSelection );
        this.add( schemaSelection );
        this.add( Box.createRigidArea(new Dimension(0,10)) );
        this.add( dropAllButton );
        this.add( Box.createRigidArea(new Dimension(0,10)) );
        this.add( createButton );
        this.add( dropButton );
        this.add( fillButton );
        
        
        // center on screen
        this.pack();
        this.setLocationRelativeTo( this.getRootPane() );
        
        this.setMinimumSize( new Dimension(this.dropAllButton.getMinimumSize().width+20, this.getPreferredSize().height) );

        this.setVisible(true);
	}
	
	private Vector<Tuple<DatabaseConfiguration,DatabaseConfiguration>> getAvailableDatabases() throws DatabaseConfigurationException {
		DatabaseConfigurationList dbList = new DatabaseConfigurationList("db.xml");
		return dbList;
	}

	@Override
	public void actionPerformed(ActionEvent arg0) {
		@SuppressWarnings("unchecked")
		Tuple<DatabaseConfiguration,DatabaseConfiguration> dbConfig = (Tuple<DatabaseConfiguration,DatabaseConfiguration>)this.dbSelection.getSelectedItem();
		DatabaseSchema schema = (DatabaseSchema)this.schemaSelection.getSelectedItem();
		
		try {
			OracleSQLDatabase appDB = new OracleSQLDatabase(dbConfig.getSecondElement());
			
			if ( arg0.getSource() == this.createButton ) {
				OracleSQLDatabase maintenanceDB = new OracleSQLDatabase(dbConfig.getFirstElement());
				schema.recreateTables(maintenanceDB, appDB);
			} else if ( arg0.getSource() == this.dropButton ) {
				OracleSQLDatabase maintenanceDB = new OracleSQLDatabase(dbConfig.getFirstElement());
				schema.dropTables(maintenanceDB, appDB);
			} else if ( arg0.getSource() == this.fillButton ) {
				NewMaintenanceDatabase maintenanceDB = NewMaintenanceDatabase.load(dbConfig.getFirstElement());
				schema.fillWithContents(maintenanceDB, appDB);
			} else if ( arg0.getSource() == this.dropAllButton ) {
				for ( String relation : appDB.getTables() ) {
					appDB.dropTable(relation);
				}
			}
		} catch (DatabaseException e) {
			e.printStackTrace();
			e.getOriginalException().printStackTrace();
		}
	}
	
	public static void main( String[] args ) throws DatabaseConfigurationException {
		new SchemaAdmin();
	}
}
