package server.parser.node;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import server.parser.Formatter.FormulaFormatter;


/**
 * Formel: <QUANTOR> <VARIABLE> (, <VARIABLE>)* <FORMULA>
 * Im ersten Kind wird die quantifizierte Formel gespeichert.
 */
public abstract class QuantifiedNode extends Node {
	private static final long serialVersionUID = -8766171801486930013L;
	private List<VariableNode> variables;

	public QuantifiedNode(List<VariableNode> quantifiedVariables, Node quantifyingFormula) {
		this.variables = new ArrayList<VariableNode>(quantifiedVariables);
		this.addChild( quantifyingFormula );
	}
	
	@Override
	public boolean isQuantifiedNode() {
		return true;
	}
	
	public List<VariableNode> getVariableNodes() {
		return this.variables;
	}
	
	/**
	 * Gibt die quantifizierte Formel zurueck. null, falls der Knoten
	 * keine Kinder hat.
	 * @return
	 */
	public Node getQuantifiedFormula() {
		try {
			return getChild(0);
		} catch ( IndexOutOfBoundsException ex ) {
			return null;
		}
	}
	
	/**
	 * Setzt die quantifizierte Formel auf einen neuen Knoten (Teilbaum).
	 * Wird als Parameter null uebergeben, so quantifiziert der Quantor nichts.
	 * Sollte im spaeteren Verlauf des Aufrufers repariert werden.
	 * @param node Ein Teilbaum, eine komplette Formel oder null.
	 */
	public void setQuantifiedFormula(Node node){
		// Kindknoten ersetzen
		this.removeAllChildren();
		if ( node != null ) {
			this.addChild(node);
		}
	}
	
	
	public abstract String getQuantor(FormulaFormatter formatter);
	
	public String toString(FormulaFormatter formulaformatter) {
		StringBuilder builder = new StringBuilder();
		builder.append(getQuantor(formulaformatter));
		builder.append(SPACE);
		String seperator = "";
		for ( VariableNode var : this.getVariableNodes() ) {
			builder.append(seperator);
			builder.append(var.toString(formulaformatter));
			seperator = formulaformatter.getQuantorSeperator(getQuantor(formulaformatter));
		}
		builder.append(SPACE);
		builder.append(getQuantifiedFormula().toString(formulaformatter));
		return builder.toString();
	}
	
	@Override
	public Set<String> getVariables() {
		Set<String> vars = this.getQuantifiedFormula().getVariables();
		
		for ( VariableNode var : this.getVariableNodes() ) {
			vars.add( var.getVariable() );
		}
		
		return vars;
	}
	
	public Set<String> getQuantifiedVariables() {
		HashSet<String> result = new HashSet<String>();
		
		for ( VariableNode var : this.getVariableNodes() ) {
			result.add( var.getVariable() );
		}
		
		return result;
	}
	
	/**
	 * Berechnet zuerst die Menge der freien Variablen im Teilbaum unter diesem Quantor.
	 * Danach werden die quantifizierten Variablen aus dieser Menge entfernt.
	 * 
	 * @return Liste aller freien Variablen unter diesem Quantor.
	 */
	@Override
	public Set<String> getFreeVariables() {
		Set<String> free = this.getQuantifiedFormula().getFreeVariables();
		free.removeAll( this.getQuantifiedVariables() );
		
		return free;
	}
	
	@Override
	public Set<String> getBoundVariables() {
		Set<String> free = this.getQuantifiedFormula().getFreeVariables();
		Set<String> bound = this.getQuantifiedVariables();
		
		// Nur dann die Variable als gebunden aufnehmen, wenn diese auch im Unterbaum
		// verwendet wird.
		bound.retainAll( free );

		// Die gebundenen Variablen der Unterbaeume hinzufuegen.
		bound.addAll( this.getQuantifiedFormula().getBoundVariables() );
		
		return bound;
	}
	
	@Override
	public boolean isElementaryFormula() {
		return true;
	}
	
	@Override
	public Set<String> rr() {
		throw new UnsupportedOperationException("Quantifizierte Formel: rr ist berechenbar nur für Exists."); 
	}
	
	@Override
	public Node flatten() {		
		while ( this.getQuantifiedFormula().getClass().equals(this.getClass()) ) {
			// Gleicher Typ (also EXISTS oder FORALL) => Variablen uebernehmen.
			QuantifiedNode quantifiedNode = (QuantifiedNode)this.getQuantifiedFormula();
			this.variables.addAll( quantifiedNode.getVariableNodes() );
			
			// Kind neu setzen (Quantor abschneiden).
			this.setQuantifiedFormula( quantifiedNode.getQuantifiedFormula() );			
		}
		
		// Sonst nur rekursiv aufrufen.
		this.setQuantifiedFormula( this.getQuantifiedFormula().flatten() );
		
		return this;
	}
	
	
	@Override
    public Node clone() {
    	QuantifiedNode copy = (QuantifiedNode)super.clone();

		// Tiefe Kopie der Variablenliste anlegen.
		ArrayList<VariableNode> newVariables = new ArrayList<VariableNode>();
		for( VariableNode var : copy.getVariableNodes() ) {
			newVariables.add( (VariableNode)var.clone() );
		}
		
		// Alte flache Kopie loeschen und neue Kopien der Kinder einfuegen.
		copy.variables = newVariables;
    	
		return copy;
    }
	

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result
				+ ((variables == null) ? 0 : variables.hashCode());
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		QuantifiedNode other = (QuantifiedNode) obj;
		if (variables == null) {
			if (other.variables != null)
				return false;
		} else if (!variables.equals(other.variables))
			return false;
		return true;
	}
	
	@Override
	public void substituteVariables(Map<String, String> mapping) {
		// durch Quantor gebundene Variablen von der Substitution ausschliessen
		Map<String, String> copiedMapping = new HashMap<String, String>(mapping);
		for( VariableNode variable : variables ) {
			copiedMapping.remove( variable.getVariable() );
		}
		super.substituteVariables(copiedMapping);
	}
	
	@Override
	public void renameVariables(Map<String, String> mapping) {
		// quantifizierte Variablen umbenennen
		for( VariableNode varNode : this.variables ) {
			varNode.renameVariables(mapping);
		}
		// evtl. nun doppelt quantifizierte Variablen entfernen
		Set<VariableNode> variableNodeSet = new HashSet<VariableNode>(this.variables);
		this.variables = new ArrayList<VariableNode>(variableNodeSet);
		
		// Umbenennung rekursiv in quantifizierter Formel fortsetzen
		this.getQuantifiedFormula().renameVariables(mapping);
	}
}
