package server.parser.node;

import java.util.HashSet;
import java.util.Set;

import server.parser.Formatter.FormulaFormatter;

public class EqualityNode extends Node {
	private static final long serialVersionUID = 513750356387646329L;
	
	/**
	 * Wenn equals = true, dann repraesentiert diese Node eine Gleichheit (z.B. X = Y),
	 * ansonsten eine Ungleichheit (z.B. X != Y).
	 */
	private boolean equals = false;
	
	public EqualityNode( TermNode left, TermNode right, boolean equals ) {
		this.addChild( left );
		this.addChild( right );
		this.equals = equals;
	}
	
	public TermNode getLeftOperand() {
		return (TermNode)this.getChild(0);
	}
	
	public TermNode getRightOperand() {
		return (TermNode)this.getChild(1);
	}
	
	/**
	 * Gibt true zurueck, wennn der Vergleich eine Gleichheit (z.B. X = Y) repraesentiert, sonst
	 * bei einer Ungleichheit (z.B. X != Y) false.
	 * 
	 * @return true, wenn X = Y, false sonst (X != Y).
	 */
	public boolean getEquals() {
		return this.equals;
	}
	
	@Override
	public boolean isEqualityNode() {
		return true;
	}
	
	/**
	 * Werteliste fuer rr():
	 * x = a oder a = x: {x}
	 * a = a, a = b oder b = a: {}
	 * x = x, x = y, y = x: {}
	 * 
	 * x != a oder a != x: {}
	 * a != a, a != b oder b != a: {}
	 * x != x: {}
	 * x != y, y != x: {}
	 */
	@Override
	public Set<String> rr() {
		Set<String> rr = new HashSet<String>();
		
		TermNode left = this.getLeftOperand();
		TermNode right = this.getRightOperand();
		
		// Bei Gleichheit und nur einer Variable wird die Variable zurueckgegeben,
		// bei zwei Konstanten automatisch die leere Menge (weil rr() von ConstantNode die leere Menge ist).
		if ( this.getEquals() ) {
			if ( (left.isConstantNode() && right.isVariableNode()) || (left.isVariableNode() && right.isConstantNode()) ) {
				// x = a oder a = x
				rr = left.rr();
				rr.addAll( right.rr() );
			}
			// Sonst: rr leer lassen.
			
			return rr;
		}
		
		// Bei Ungleichheit die leere Menge zurueckgeben.
		
		return rr;
	}

	@Override
	public String toString(FormulaFormatter formulaformatter) {
		String result = "";
		
		// Teil links des Gleichheitszeichens.
		result += this.getLeftOperand().toString(formulaformatter);
		// Das (Un-)Gleichheitszeichen selbst.
		if ( this.getEquals() ) {
			result += formulaformatter.getEquals();
		} else {
			result += formulaformatter.getNotEquals();
		}
		// Teil rechts des Gleichheitszeichens.
		result += this.getRightOperand().toString(formulaformatter);
		
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + (equals ? 1231 : 1237);
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		EqualityNode other = (EqualityNode) obj;
		if (equals != other.equals)
			return false;
		return true;
	}
	
	@Override
	public Set<String> getFreeVariables() {
		Set<String> free = new HashSet<String>();
		
		if( this.getLeftOperand().isVariableNode() )
			free.add( ((VariableNode)this.getLeftOperand()).getVariable() );
		if( this.getRightOperand().isVariableNode() )
			free.add( ((VariableNode)this.getRightOperand()).getVariable() );
		
		return free;
	}
	
	@Override
	protected void pushNegations( boolean foundNegation, boolean skipNegatedExistentialQuantifiers ) {
		// Wir sind in einem Blatt des Baums.
		
		if ( foundNegation ) {
			// Es wurde vorher eine Negation gefunden, d.h. die (Un)Gleichheit muss negiert werden.
			this.equals = !this.equals;
		}
		
		// In jedem Fall sind wir an dieser Stelle fertig (kein rekursiver Aufruf).
	}
}
