package server.parser;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import server.parser.FormulaUtil;
import server.parser.node.ConjunctorNode;
import server.parser.node.DisjunctorNode;
import server.parser.node.ExistentialQuantifiedNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.QuantifiedNode;

import exception.UnsupportedFormulaException;

/**
 * RANF-ALG (Relational Algebra Normal Form), siehe Abiteboul S.88
 * Umformung einer Formel, welche in SRNF vorliegt, in eine dazu äquivalente Formel in RANF
 */
public class FormulaToRANF {
	
	/**
	 * Berechnet rekursiv die Potenzmenge einer Menge. Die Laufzeit ist O(n^2).
	 * @param <T> Typ der Menge.
	 * @param originalSet Menge, von der die Potenzmenge berechnet werden soll.
	 * @return Potenzmenge der Menge.
	 */
	private static <T> Set<Set<T>> powerSet( Set<T> originalSet ) {
	    Set<Set<T>> sets = new HashSet<Set<T>>();
	    if (originalSet.isEmpty()) {
	        sets.add(new HashSet<T>());
	        return sets;
	    }
	    List<T> list = new ArrayList<T>(originalSet);
	    T head = list.get(0);
	    Set<T> rest = new HashSet<T>(list.subList(1, list.size())); 
	    for (Set<T> set : powerSet(rest)) {
	        Set<T> newSet = new HashSet<T>();
	        newSet.add(head);
	        newSet.addAll(set);
	        sets.add(newSet);
	        sets.add(set);
	    }           
	    return sets;
	}
	
	public static void RANF( Formula formula ) throws UnsupportedFormulaException {
		Node newRoot = FormulaToRANF.RANF( formula.getRootChild() );
		formula.setRootChild( newRoot );
	}
	
	private static Node RANF( Node root ) throws UnsupportedFormulaException {
		// RANF-Faelle nur anwendbar, wenn der Knoten einer Konjunktion entspricht
		// ansonsten kann die Knoten (die Formel) unveraendert zurueckgegeben werden
		if ( root.isConjunctorNode() ) {
			Node xi = null;
			ConjunctorNode phi = new ConjunctorNode();
			ExistentialQuantifiedNode existsXi = null;
			ExistentialQuantifiedNode negatedExistsXi = null;
			
			// RANF_R1			
			// Pruefe ob die Formel den folgenden Aufbau hat: phi_1 & phi_2 & ... & phi_n & (xi_1 v xi_2 v ... v xi_m)
			phi = new ConjunctorNode();
			for ( Node operand: ((ConjunctorNode)root).getOperands() ) {
				// Teilformel xi = (xi_1 v xi_2 v ... v xi_m) vorhanden?
				if ( operand.isDisjunctorNode() && xi == null ) {
					xi = operand;
				} else {
					// restliche Teilformeln (Phi's)
					phi.addChild( operand );
				}
			}
						
			// Vorbedingung pruefen: rr(xi) != free(xi)
			// ist Xi null bedeutet dies, dass keine Teilformel mit disjunktiv verknuepften Teilformeln gefunden wurde und der
			// Fall nicht zu trifft
			if ( xi != null && !FormulaUtil.rangerestricted(xi) ) {
				// Kindknoten von root aus Kindliste entfernen, damit Aufrufer mit der durch die Umformung zerstoerten Formel root keinen Schaden anrichten kann
				root.removeAllChildren();
				return RANF_R1(phi, (DisjunctorNode) xi);
			} 	
			
			// RANF_R2
			// Pruefe ob die Formel den folgenden Aufbau hat: phi_1 & phi_2 & ... & phi_n & EXISTS <VAR> xi		
			phi = new ConjunctorNode();
			for ( Node operand: ((ConjunctorNode)root).getOperands() ) {
				// Teilformel EXISTS <VAR> xi vorhanden?
				if ( operand.isExistentialQuantifiedNode() && xi == null ) {
					existsXi = (ExistentialQuantifiedNode)operand;
					xi = existsXi.getQuantifiedFormula();
				} else {
					// alle nicht quantifizierten Subformeln (Phi's)
					phi.addChild(operand);
				}
			}
			
			// Vorbedingung pruefen: rr(xi) != free(xi)
			// ist Xi null bedeutet dies, dass keine quantifizierte Teilformel mit disjunktiv verknuepften Teilformeln gefunden wurde und der
			// Fall nicht zu trifft
			if ( xi != null && !(FormulaUtil.rangerestricted(xi)) ) {
				// Kindknoten von root aus Kindliste entfernen, damit Aufrufer mit der durch die Umformung zerstoerten Formel root keinen Schaden anrichten kann
				root.removeAllChildren();
				return RANF_R2( phi, existsXi ); 
			}	
			
			// RANF_R3
			// Pruefe, ob die Formel den folgenden Aufbau hat: phi_1 & phi_2 & ... & phi_n & NOT EXISTS <VAR> xi
			phi = new ConjunctorNode();
			for ( Node operand: ((ConjunctorNode)root).getOperands() ) {
				// Teilformel NOT EXISTS <VAR> xi vorhanden?
				if ( operand.isNegationNode() && ((NegationNode)operand).getNegatedFormula().isExistentialQuantifiedNode() && xi == null ) {
					negatedExistsXi = (ExistentialQuantifiedNode) ((NegationNode)operand).getNegatedFormula();
					xi = negatedExistsXi.getQuantifiedFormula();
				} else {
					// alle nicht negierten und quantifizierten Subformeln (Phi's)
					phi.addChild( operand );
				}
			}
			
			// Vorbedingung fuer pruefen: rr(xi) != free(xi)
			if ( xi != null && !(FormulaUtil.rangerestricted(xi)) ) {
				// Kindknoten von root aus Kindliste entfernen, damit Aufrufer mit der durch die Umformung zerstoerten Formel root keinen Schaden anrichten kann
				root.removeAllChildren();
				return RANF_R3( phi, negatedExistsXi );
			}
			
		} else {
			// RANF rekursiv auf alle Kinder anwenden
			ArrayList<Node> newChildList = new ArrayList<Node>();
			
			for ( Node child : root.getChildren() ) {
				newChildList.add( RANF(child) );
			}
			
			root.replaceChildren( newChildList );
						
			return root;			
		}
		
		return root;
	}	
	
	// phi_1 & phi_2 & ... & phi_n & (xi_1 v xi_2 v ... v xi_m)
	private static Node RANF_R1 ( ConjunctorNode phi, DisjunctorNode xi ) throws UnsupportedFormulaException {
		// Alpha wird nicht benoetigt (Sonderfall!)
		Node beta = null;	
		DisjunctorNode xi_apostrophe = new DisjunctorNode();	
		List<Node> phis = new ArrayList<Node>();		

		// Kinder der uebergebenen Konjunktion auslesen
		phis = phi.getChildren();
		
		
		// fuer JEDEN Operanden von Xi JEWEILS Konjunktion mit ALLEN Phi's aufbauen (Sonderfall!)
		// fuer jeden Operanden xi_i: (xi_i AND phi_1 AND ... AND phi_n)
		for ( Node xi_i : xi.getOperands() ) {		
			ConjunctorNode conjunction = new ConjunctorNode();
			// xi_i clonen um den alten Syntaxbaum von xi nicht waehrend der Iteration zu modifizieren
			conjunction.addChild( xi_i.clone() );
			// alle phis clonen, da diese in mehreren Teilformeln vorkommen
			for( Node phi_i : phis ) {
				conjunction.addChild( phi_i.clone() );
			}
			xi_apostrophe.addChild( conjunction );			
		}
		
		// Beta zusammenbauen: RANF((xi_1 AND phi_1 AND ... AND phi_n) OR .. OR (xi_m AND phi_1 AND ... AND phi_n))
		FormulaToSRNF.srnf(xi_apostrophe);
		beta = RANF( xi_apostrophe );
		
		return beta;
		
	}
	
	// phi_1 & phi_2 & ... & phi_n & EXISTS <VAR> xi
	private static Node RANF_R2 ( ConjunctorNode phi, ExistentialQuantifiedNode existsXi ) throws UnsupportedFormulaException {
		Node alpha = null;
		Node beta = null;
		Node xi = null;
		ConjunctorNode xi_apostrophe = null;
		HashSet<Node> phis = new HashSet<Node>();		
		
		// FIXME: StackOverflowError (hashCode())
		// Kinder der uebergebenen Konjunktion auslesen
		phis.addAll( phi.getChildren() );
		
		// quantifizierte Teilformel aus der uebergebenen existenzquantifizierten Formel auslesen
		// und clonen, da diese spaeter auch Teil einer anderen Formel wird
		xi = ( (QuantifiedNode)existsXi ).getQuantifiedFormula().clone();
		
		// Berechne die Potenzmenge der phis und suche ein Element, so dass die
		// Konjunktion zusammen mit dem xi rangerestricted ist
		Set<Set<Node>> powerSet = FormulaToRANF.powerSet( phis );
		for ( Set<Node> subset : powerSet ) {
			ConjunctorNode conjunction = new ConjunctorNode();
			conjunction.addChildren( subset );
			conjunction.addChild( xi );
			if ( FormulaUtil.rangerestricted(conjunction) ) {
				// Xi' entspricht der neuen quantifizierten Teilformel
				// (phi_i_1 AND ... AND phi_i_k AND xi)
				xi_apostrophe = conjunction;
				// die Menge Phis ergibt sich schliesslich zu der Menge aller Phis, die nicht in Xi' verwendet wurden
				phis.removeAll( subset );
				break;
			}
		}
		
		// Alpha zusammenbauen: RANF(phi_j_1 AND ... AND phi_j_l)
		// phis = (phi_j_1 AND ... AND phi_j_l), Menge aller noch nicht verwendeten Phis (s.oben)
		ConjunctorNode alpha_tmp = new ConjunctorNode();
		// clonen der phis nicht erforderlich, da es genau jene sind, die _nicht_ in xi_apostrophe enthalten sind
		alpha_tmp.addChildren( phis );
		alpha = RANF( alpha_tmp );
	
		// Beta zusammenbauen: RANF(SRNF((phi_i_1 AND ... AND phi_i_k AND xi)))
		FormulaToSRNF.srnf(xi_apostrophe);
		beta = RANF( xi_apostrophe );
		
		// Phi' zusammenbauen: Alpha AND EXISTS <VAR> BETA
		alpha.addChild( new ExistentialQuantifiedNode(existsXi.getVariableNodes(), beta) );
		
		return alpha;
	}
	
	private static Node RANF_R3 ( ConjunctorNode phi, ExistentialQuantifiedNode existsXi ) throws UnsupportedFormulaException {
		Node alpha = null;
		Node beta = null;
		Node xi = null;
		ConjunctorNode xi_apostrophe = null;
		HashSet<Node> phis = new HashSet<Node>();

		// quantifizierte Teilformel aus der uebergebenen existenzquantifizierten Formel auslesen
		// und clonen, da diese spaeter auch Teil einer anderen Formel wird
		xi = ( (QuantifiedNode)existsXi ).getQuantifiedFormula().clone();
		
		// Kinder der uebergebenen Konjunktion auslesen
		phis.addAll( phi.getChildren() );
				
		// Berechne die Potenzmenge der phis und suche ein Element, so dass die
		// Konjunktion zusammen mit dem xi rangerestricted ist.
		Set<Set<Node>> powerSet = FormulaToRANF.powerSet( phis );
		for ( Set<Node> subset : powerSet ) {
			ConjunctorNode conjunction = new ConjunctorNode();
			conjunction.addChildren( subset );
			conjunction.addChild( xi );
			if ( FormulaUtil.rangerestricted(conjunction) ) {
				// Xi' entspricht der neuen (negativ) quantifizierten Teilformel
				// (phi_i_1 AND ... AND phi_i_k AND xi)
				xi_apostrophe = conjunction;
				break;
			}
		}
		
		// Alpha zusammenbauen: RANF(phi_1 AND ... AND phi_n)
		ConjunctorNode alpha_tmp = new ConjunctorNode();
		// phis clonen, da ein Teil von ihnen bereits in xi_apostrophe enthalten ist
		for( Node phi_i : phis ) {
			alpha_tmp.addChild(phi_i.clone());
		}
		alpha = RANF( alpha_tmp );
	
		// Beta zusammenbauen: RANF(SRNF((phi_i_1 AND ... AND phi_i_k AND xi)))
		FormulaToSRNF.srnf(xi_apostrophe);
		beta = RANF( xi_apostrophe );
		
		// Phi' zusammenbauen: Alpha AND NOT EXISTS <VAR> BETA
		alpha.addChild( new NegationNode (new ExistentialQuantifiedNode(existsXi.getVariableNodes(), beta)) );
		
		return alpha;
		
	}
	
	public static void modifiedRANF( Formula formula ) {
		Node newRoot = FormulaToRANF.modifiedRANF( formula.getRootChild() );
		formula.setRootChild( newRoot );
	}
	
	/** Realisiert die Umformung einer Formel, welche in RANF vorliegt, in die modifizierte RANF. (s. Abiteboul Seite 88) 
	 * 
	 * @param Node Formel in RANF
	 * @return Node Formel in modified RANF
	 */
	private static Node modifiedRANF( Node root ) {
		if( root.isAtomPredicateNode() || root.isEqualityNode() ) {
			// ist trivialerweise schon in modified RANF
			return root;
		}
		
		ArrayList<Node> newChildList = new ArrayList<Node>();
		List<Node> childrenEquality = new ArrayList<Node>();
		List<Node> childrenNeg = new ArrayList<Node>();
		ConjunctorNode conjunc = new ConjunctorNode();
		
		// modifiedRANF auf alle Kinder anwenden
		for ( Node child : root.getChildren()) {
			newChildList.add( modifiedRANF( child ));
		}
		root.replaceChildren(newChildList);
		
		
		// nur Konjunktionen mit mehr als zwei Kindern betrachten
		if ( root.isConjunctorNode() && root.getChildrenNumber() > 2 ) {
			
			for ( Node child : root.getChildren() ) {
				// Negationsknoten identifizieren
				if ( child.isNegationNode() ) {
					childrenNeg.add( child );
				}
				// Gleichheitsknoten identifizieren
				else if ( child.isEqualityNode() ) {
					childrenEquality.add( child );
				}
				// alle anderen Knoten
				else {					
					// Konjunktion hat bereits zwei Kinder, weswegen ein neuer Konjunktionsknoten erzeugt wird und die bereits bestehende
					// Konjunktion an diesen angehaengt wird
					if ( conjunc.getChildrenNumber() == 2 ) {
						ConjunctorNode conjunctor = new ConjunctorNode();
						conjunctor.addChild( conjunc );
						conjunctor.addChild( child );					
						conjunc = conjunctor;
					}
					
					// Konjunktion hat noch keine zwei Kinder, weswegen der aktuelle Knoten an die Konjunktion angehängt werden kann ohne das
					// ein neuer Konjunktionsknoten erzeugt werden muss
					else {
						conjunc.addChild( child );							
					}					
				
				}

			}
			
			// identifizierte Negationsknoten anhaengen; für jeden dieser Knoten wird ein neuer Konjunktionsknoten erzeugt
			for ( Node childNeg : childrenNeg ) {
				ConjunctorNode conjunctor = new ConjunctorNode();
				
				if ( conjunc.getChildrenNumber() < 2 ) {
					conjunc.addChild( childNeg );					
				}				
				else {
					conjunctor.addChild( conjunc );
					conjunctor.addChild( childNeg );
					conjunc = conjunctor;
				}
			
			}
			
			// identifizierte Gleichheitsknoten anhaengen; für jeden dieser Knoten wird ein neuer Konjunktionsknoten erzeugt
			for ( Node childEquality : childrenEquality ) {
				ConjunctorNode conjunctor = new ConjunctorNode();
				
				if ( conjunc.getChildrenNumber() < 2 ) {
					conjunc.addChild( childEquality );					
				}				
				else {
					conjunctor.addChild( conjunc );
					conjunctor.addChild( childEquality );
					conjunc = conjunctor;		
				}
	
			}
				
		} 
	
//		// binaere AND's identifizieren und eventuell umsortieren
//		else if ( root.isConjunctorNode() && root.getChildrenNumber() == 2 ) {
//			
//			// Negation auch bei binaeren AND's lediglich auf linker Seite erlaubt
//			// Ausnahme: auf rechter Seite Equality (BSP: NOT beinbruch(X) AND X="dirk")
//			if ( root.getChild(0).isNegationNode() && !root.getChild(1).isEqualityNode() ) {
//				List<Node> rootChildren = new ArrayList<Node>();
//				rootChildren.add( root.getChild(1) );
//				rootChildren.add( root.getChild(0) );
//				root.setChildren( rootChildren );				
//				return root;		
//			} 
//		
//			// Equality auch bei binaeren AND's lediglich auf rechter Seite erlaubt
//			if ( root.getChild(0).isEqualityNode() ) {
//				List<Node> rootChildren = new ArrayList<Node>();
//				rootChildren.add( root.getChild(1) );
//				rootChildren.add( root.getChild(0) );
//				root.setChildren( rootChildren );			
//				return root;
//			}
//			
//			return root;
//
//		}
		
		// Knoten ist keine Konjunktion oder eine Konjunktion mit zwei Kindern, wobei Equality und Negation nur auf rechter
		// Seite vorkommen (Ausnahme: links Negation, rechts Equality)
		else {
			return root;
		}
		return conjunc;		
	}

}

