package server.database.sql;

import java.util.LinkedList;
import java.util.List;
import java.util.Vector;
import java.util.regex.PatternSyntaxException;

import org.apache.log4j.Logger;

import server.database.schema.SchemaColumn;
import server.database.schema.TableSchema;
import server.database.sql.SQLWhereClause.Condition;
import server.util.Tuple;

/**
 * Speichert die Komponenten der aktuellen Anfrage zwischen. Aus den einzelnen Komponenten wird
 * dann ein gueltiger SQL-String aufgebaut.
 *
 */
public class SQLExpression {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");
	
	private Vector<String> select;
	private boolean distinct;
	private String from;
	private String whereOverride;
	private Vector<SQLWhereClause> where;
	private Vector<Object> whereVariables;
	private Vector<Tuple<String,Boolean>> orderBy;
	private Vector<String> groupBy;
	private Vector<Tuple<String,Vector<Tuple<String,String>>>> join;
	private int limit;
	private List<Tuple<String, String>> columnValues;
	private Vector<Object> setVariables;
	private Vector<String> generatedColumns;
	
	/**
	 * Erzeugt eine neue SQLExpression, der aus einer leeren SQL-Anfrage besteht (kann durch die einzelnen Methoden
	 * veraendert werden) und ein leeres ResultSet (null) hat.
	 */
	public SQLExpression() {
		super();
		
		this.select = new Vector<String>();
		this.distinct = false;
		this.from = "";
		this.whereOverride = null;
		this.where = new Vector<SQLWhereClause>();
		this.whereVariables = new Vector<Object>();
		this.orderBy = new Vector<Tuple<String,Boolean>>();
		this.groupBy = new Vector<String>();
		this.join = new Vector<Tuple<String,Vector<Tuple<String,String>>>>();
		this.limit = -1;
		this.columnValues = new LinkedList<Tuple<String, String>>();
		this.setVariables = new Vector<Object>();
		this.generatedColumns = new Vector<String>();
	}
	
	/**
	 * Gibt den SELECT-Teil des SQL-Query zurueck. Jedes einzelne Element der Liste entspricht einem normalerweise durch
	 * Komma getrennten Teil des SELECT-Parameters in SQL (z.B. bei "SELECT a,b FROM ..." waeren die Elemente "a" und "b" enthalten).
	 * @return SELECT-Parameter einer SQL-Anfrage als Liste.
	 */
	public Vector<String> getSelect() {
		return select;
	}
	
	public SQLExpression select( SchemaColumn column ) {
		this.select.add( column.name );
		
		return this;
	}
	
	public SQLExpression select( String column ) {
		String[] split = null;
		
		try {
			split = column.split( "," );
			
			for ( String col : split ) {
				this.select.add( col );
			}
		} catch ( PatternSyntaxException exception ) {
			// This one should be impossible. The regex pattern is always valid.
			logger.fatal("Invalid regex pattern in select(). This is impossible. Check your code.", exception);
			return this;
		}
				
		return this;
	}
	
	
	public boolean isDistinct() {
		return distinct;
	}
	
	public SQLExpression distinct() {
		this.distinct = true;
		
		return this;
	}
	

	public Vector<Tuple<String,Boolean>> getOrderBy() {
		return orderBy;
	}
	
	public SQLExpression order_by( SchemaColumn column ) {
		return this.order_by( column, false );
	}
	
	public SQLExpression order_by( SchemaColumn column, boolean desc ) {
		this.orderBy.add( new Tuple<String,Boolean>(column.name, desc) );
		
		return this;
	}
	
	public SQLExpression order_by( String column ) {
		return this.order_by( column, false );
	}
	
	public SQLExpression order_by( String column, boolean desc ) {
		this.orderBy.add( new Tuple<String,Boolean>(column, desc) );
		
		return this;
	}
	
	
	public Vector<String> getGroupBy() {
		return groupBy;
	}
	
	public SQLExpression group_by( String column ) {
		this.groupBy.add( column );
		
		return this;
	}
	
	
	public Vector<Tuple<String, Vector<Tuple<String,String>>>> getJoin() {
		return join;
	}
	
	/**
	 * @deprecated
	 * @param table
	 * @param on
	 * @return
	 */
	public SQLExpression join( String table, Vector<Tuple<String,String>> on ) {
		this.join.add( new Tuple<String,Vector<Tuple<String,String>>>(table, on) );
		
		return this;
	}
	
	public SQLExpression join( TableSchema table, Vector<Tuple<String,String>> on ) {
		this.join.add( new Tuple<String,Vector<Tuple<String,String>>>(table.name, on) );
		
		return this;
	}
	
	
	public int getLimit() {
		return limit;
	}

	public SQLExpression limit( int number ) {
		this.limit = number;
		
		return this;
	}
	
	
	public List<Tuple<String, String>> getColumnValues() {
		return columnValues;
	}
	
	/**
	 * Legt den Wert einer Spalte fuer ein INSERT oder UPDATE fest.
	 * @param column Spalte, fuer den der Wert festgelegt werden soll.
	 * @param value Beliebiger String.
	 * @return
	 */
	public SQLExpression set(SchemaColumn column, String value) {
		return this.set( column, value, false );
	}
	
	/**
	 * Legt den Wert einer Spalte fuer ein INSERT oder UPDATE fest.
	 * @param column Spalte, fuer den der Wert festgelegt werden soll.
	 * @param number Beliebige Zahl.
	 * @return
	 */
	public SQLExpression set(SchemaColumn column, int number) {
		return this.set( column, String.valueOf(number), true );
	}
	
	/**
	 * Legt den Wert einer Spalte fuer ein INSERT oder UPDATE fest.
	 * @param column Spalte, fuer den der Wert festgelegt werden soll.
	 * @param value Beliebiger String.
	 * @param noprime Falls true werden um value KEINE einfachen Anfuehrungszeichen gesetzt.
	 * @return
	 */
	public SQLExpression set(SchemaColumn column, String value, boolean noprime) {
		return this.set( column.name, value, noprime );
	}
	
	/**
	 * Legt den Wert einer Spalte fuer ein INSERT oder UPDATE fest.
	 * @param column Spalte, fuer den der Wert festgelegt werden soll.
	 * @param value Beliebige Zahl.
	 * @return
	 * @deprecated
	 */
	public SQLExpression set(String column, int number){
		return this.set( column, String.valueOf(number), true );
	}
	
	/**
	 * Legt den Wert einer Spalte fuer ein INSERT oder UPDATE fest.
	 * @param column Spalte, fuer den der Wert festgelegt werden soll.
	 * @param value Beliebiger String.
	 * @return
	 * @deprecated
	 */
	public SQLExpression set(String column, String value){
		return this.set( column, value, false );
	}
	
	/**
	 * FIXME: merge with method above.
	 * @param column
	 * @param value
	 * @param noprime
	 * @return
	 * @deprecated
	 */
	public SQLExpression set(String column, String value, boolean noprime ){
		// Variable fuer bind-Variable festlegen.
		this.getSetVariables().add( value );
		
		if ( !noprime ) {
			value = "'"+value+"'";
		}

		this.columnValues.add( new Tuple<String, String>(column, value) );
		
		return this;
	}
	
	
	public Vector<String> getGeneratedColumns() {
		return this.generatedColumns;
	}
	
	/**
	 * Mit dieser Methode wird festgelegt, welche Spalten bei einem INSERT automatisch von der Datenbank
	 * generiert werden.
	 * @param column
	 */
	public void setGenerated( String column ) {
		this.generatedColumns.add( column );
	}
	
	
	/**
	 * Gibt die Tabelle zurueck, auf den die SQL-Anfrage gestellt werden soll. Entspricht dem Parameter hinter FROM in SQL.
	 * @return FROM-Parameter einer SQL-Anfrage.
	 */
	public String getFrom() {
		return from;
	}
	
	/**
	 * @param from
	 * @return
	 */
	public SQLExpression from(String from) {
		this.from = from;
		return this;
	}
	
	public SQLExpression from(TableSchema table) {
		this.from = table.name;
		return this;
	}
	
	/**
	 * Gibt die Variablen zurueck, die fuer die bind-Variablen in den SQL-Query eingesetzt werden muessen.
	 * @return Liste von Elementen, die fuer bind-Variablen eingesetzt werden muessen.
	 */
	public Vector<Object> getVars() {
		Vector<Object> vars = new Vector<Object>();
		vars.addAll( this.getSetVariables() );
		vars.addAll( this.getWhereVariables() );
		
		return vars;
	}
	
	public Vector<Object> getWhereVariables() {
		return this.whereVariables;
	}
	
	public void setWhereVariables( Vector<Object> whereVariables ) {
		this.whereVariables = whereVariables;
	}
	
	public Vector<Object> getSetVariables() {
		return this.setVariables;
	}
	
	/**
	 * Gibt alle WHERE-Bedingungen 
	 * @return
	 */
	public Vector<SQLWhereClause> getWhere() {
		return where;
	}
	
	/**
	 * Allows you to override the other where() command and specify your own where clause.
	 * Please use bind variable, which can be set with the method {@link #setWhereVariables(Vector)}.
	 * @param whereClause WHERE Expression (must be valid SQL).
	 * @return
	 */
	public SQLExpression where( String whereClause ) {
		this.whereOverride = whereClause;
		return this;
	}
	
	public SQLExpression where( String column, String value ) {
		return this.where( column, value, Condition.EQUAL, Condition.AND, false );
	}
	
	public SQLExpression where(String column, int number) {
		return this.where( column, String.valueOf(number), Condition.EQUAL, Condition.AND, true );
	}
	
	public SQLExpression where( String column, String value, Condition comparison ) {
		return this.where( column, value, comparison, Condition.AND, false );
	}
	
	public SQLExpression where( String column, int number, Condition comparison ) {
		return this.where( column, String.valueOf(number), comparison, Condition.AND, true );
	}
	
	public SQLExpression or_where( String column, String value ) {
		return this.where( column, value, Condition.EQUAL, Condition.OR, false );
	}
	
	public SQLExpression or_where(String column, int number) {
		return this.where( column, String.valueOf(number), Condition.EQUAL, Condition.OR, true );
	}
	
	public SQLExpression or_where( String column, String value, Condition comparison ) {
		return this.where( column, value, comparison, Condition.OR, false );
	}
	
	public SQLExpression or_where( String column, int number, Condition comparison ) {
		return this.where( column, String.valueOf(number), comparison, Condition.OR, true );
	}
	
	/**
	 * Genau wie where(), allerdings wird bei noprime = true der zweite Wert nicht in einfache Anführungszeichen
	 * (') gesetzt. Wird z.B. für den Join von Tabellen benutzt.
	 * @param column
	 * @param value
	 * @param noprime
	 * @return
	 */
	public SQLExpression where( String column, String value, Condition comparison, Condition connective, boolean noprime ) {
		// Variable fuer bind-Variable festlegen.
		this.getWhereVariables().add( value );

		if ( !noprime ) {
			value = "'"+value+"'";
		}
		
		this.getWhere().add( new SQLWhereClause(column, value, comparison, connective) );
	
		return this;
	}
	
	
	/**
	 * Gibt die SQL-Anfrage als String zurueck. Dabei werden die Werte direkt eingesetzt und keine bind-Variablen verwendet.
	 * @return SQL-Query als String-Darstellung.
	 */
	public String getSelectSQL() {
		return this.getSelectSQL( false );
	}
	
	public String getSelectSQL( boolean bindVars ) {
		String result = "";
		String sep = "";
		
		// SELECT ..
		result = "SELECT ";
		
		if ( this.distinct ) {
			result += "DISTINCT ";
		}
		
		if ( this.select.size() == 0 ) {
			result += "*";
		} else {
			for ( String column : this.select ) {
				result += sep + column;
				sep = ",";
			}
		}
		
		// FROM ..
		result += " FROM "+this.getFrom();
		
		// INNER/NATURAL JOIN ..
		for ( Tuple<String,Vector<Tuple<String,String>>> join : this.join ) {
			if ( join.getSecondElement() == null ) {
				// NATURAL JOIN
				result += " NATURAL JOIN "+join.getFirstElement();
			}
			else
			{
				// INNER JOIN
				result += " INNER JOIN "+join.getFirstElement() + " ON (";
				
				sep = "";
				for ( Tuple<String,String> on : join.getSecondElement() ) {
					result += on.getFirstElement() + " = " + on.getSecondElement();
					sep = ",";
				}
				
				result += ")";
			}
		}
		
		// WHERE ..
		result += this.generateWhereSQL( bindVars );
		
		// ORDER BY
		if ( this.orderBy.size() > 0 ) {
			result += " ORDER BY ";
			
			sep = "";
			for ( Tuple<String,Boolean> column : this.orderBy ) {
				result += sep + column.getFirstElement();
				if ( column.getSecondElement() ) {
					result += " DESC";
				}
				sep = ",";
			}
		}
		
		// GROUP BY
		if ( this.groupBy.size() > 0 ) {
			result += " GROUP BY ";
			
			sep = "";
			for ( String column : this.groupBy) {
				result += sep + column;
				sep = ",";
			}
		}
		
		// LIMIT (nur mySQL)
		if ( this.limit > 0 ) {
			result += " LIMIT "+this.limit;
		}
		
		return result;
	}
	
	public String getInsertSQL( String table ) {
		return this.getInsertSQL( table, false );
	}
	
	public String getInsertSQL( String table, boolean bindVars ) {
		String columns = "";
		String values = "";		
		String sep = "";
		
		// NULL-Pointer zurückgeben, wenn keine columnValues angegeben wurden.
		if ( this.columnValues.isEmpty() ) {
			return null;
		}
		
		for ( Tuple<String, String> columnValue : this.columnValues ) {
			columns += sep + columnValue.getFirstElement();
			if ( bindVars ) {
				values += sep + "?";
			} else {
				values += sep + columnValue.getSecondElement();
			}
			
			sep = ",";
		}
				
		return "INSERT INTO " + table + "(" + columns + ") VALUES (" + values + ")";
	}
	
	
	/**
	 * Gibt die SQL-Anfrage als String zurueck. Dabei werden die Werte direkt eingesetzt und keine bind-Variablen verwendet.
	 * @return SQL-Update als String-Darstellung.
	 */
	public String getUpdateSQL( String table ) {
		return this.getUpdateSQL(table, false );
	}
	
	public String getUpdateSQL( TableSchema table ) {
		return this.getUpdateSQL(table, false );
	}
	
	public String getUpdateSQL( TableSchema table, boolean bindVars ) {
		return this.getUpdateSQL(table.name, bindVars );
	}
	
	public String getUpdateSQL( String table, boolean bindVars ) {
		String result = "";
		String sep = "";
		
		// NULL-Pointer zurückgeben, wenn keine columnValues angegeben wurden.
		if ( this.columnValues.isEmpty() ) {
			return null;
		}
		
		// UPDATE ..
		result = "UPDATE " + table;
		
		// SET ..
		result += " SET ";
		for ( Tuple<String, String> columnValue : this.columnValues ) {
			result += sep + columnValue.getFirstElement() + "=";
			if ( bindVars ) {
				result += "?";
			} else {
				result += columnValue.getSecondElement();
			}
			sep = ",";
		}
				
		// WHERE ..
		result += this.generateWhereSQL( bindVars );
		
		return result;
	}
	
	
	public String getDeleteSQL( String table ) {
		return this.getDeleteSQL( table, false );
	}
	
	public String getDeleteSQL( String table, boolean bindVars ) {
		String result = "";
		
		// DELETE FROM ..
		result = "DELETE FROM "+table;
		
		// WHERE ..
		result += this.generateWhereSQL( bindVars );
		
		return result;
	}
	
	/**
	 * Gibt den WHERE-Teil von SQL-Ausdruecken wie SELECT, DELETE oder UPDATE zurueck.
	 * @param bindVars true, falls bind-Variablen (Fragezeichen im SQL-Ausdruck) verwendet werden sollen, ansonsten false.
	 * @return SQL-WHERE-Klausel eines SQL-Ausdruck.
	 */
	private String generateWhereSQL( boolean bindVars ) {
		String result = "";
		
		// SQL wurde extra spezifiziert, dieses benutzen.
		if ( whereOverride != null ) {
			return " WHERE "+this.whereOverride;
		}
		
		if ( this.getWhere().size() > 0 ) {
			result += " WHERE ";
			
			boolean firstrun = true;
			for ( SQLWhereClause tupel : this.getWhere() ) {
				if ( !firstrun ) {
					result += tupel.getConnective();
				}
				result += tupel.getLeftOperand() + tupel.getComparison();
				if ( bindVars ) {
					result += "?";
				} else {
					result += tupel.getRightOperand();
				}
				
				firstrun = false;
			}
		}
		
		return result;
	}
}
