package server.database.sql;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.apache.log4j.Logger;

import server.database.DatabaseConfiguration;
import server.database.schema.TableSchema;
import server.database.sql.SQLWhereClause.Condition;
import server.util.Tuple;

import exception.DatabaseException;

/**
 * Abstrakte Klasse zur Implementierung einer konkreten SQL-Datenbankanbindung.
 * 
 * Alle Methoden zum Aufbau eines SQL-Queries geben als Rueckgabewert die
 * aktuelle Instanz des Datenbank-Objekts zurueck. Dadurch kann man mittels
 * Method-Chaining relativ uebersichtlich Queries aufbaun.
 * Beispiel: db.select(*).from("test").where("id", "10");

 * Nach dem Ausführen von query() wird eine neuer leerer SQL-Query erstellt.
 * @author Dirk Schalge
 *
 */
public abstract class SQLDatabase {
	protected final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.database");
	
	protected Connection connection;
	private SQLExpression currentQuery;
	// Stores the schema information of this database for fast access.
	private Map<String, List<Column>> dataCache;
	
	protected SQLDatabase( DatabaseConfiguration config ) throws DatabaseException {
		// Establish database connection.
		try {
			this.connect( config );
		} catch ( SQLException e ) {
			throw new DatabaseException( "Unable to connect to database", e );
		}
		
		// Load schema information.
		this.reloadSchemaInformation();
		
		// Old
		this.currentQuery = new SQLExpression();
	}
	
	protected abstract void connect( DatabaseConfiguration config ) throws SQLException;
	public abstract void lockShared( String table ) throws DatabaseException;
	public abstract void lockExclusive( String table ) throws DatabaseException;
	public abstract void commit() throws DatabaseException;
	public abstract void rollback() throws DatabaseException;
	
	
	
	/* #############################
	 * New experimental stuff below!
	 * #############################
	 */
	
	/**
	 * Returns the current connection used to communicate with the database.
	 * @return Connection to the database.
	 */
	public Connection getDatabaseConnection() {
		return this.connection;
	}
	
	public SQLSelect newSelect() {
		return new SQLSelect( this );
	}
	
	public SQLInsert newInsert() {
		return new SQLInsert( this );
	}
	
	public SQLUpdate newUpdate() {
		return new SQLUpdate( this );
	}
	
	public SQLDelete newDelete() {
		return new SQLDelete( this );
	}
	
	
	/* ################
	 * Old stuff below.
	 * ################
	 */
	
	@Deprecated
	public SQLExpression getCurrentQuery() {
		return currentQuery;
	}

	@Deprecated
	public void setCurrentQuery(SQLExpression currentQuery) {
		this.currentQuery = currentQuery;
	}

	
	/**
	 * Startet einen neuen SQL-Ausdruck und gibt diesen zur weiteren Manipulation zurueck.
	 * Aufrufe von get(), insert(), update() und delete() verwenden den zuletzt mittels
	 * sql() erzeugten SQL-Ausdruck. Nach Aufruf einer dieser Methoden kann die letzt
	 * SQL-Anfrage mittels getLastQuery() abgefragt werden.
	 * @return Leerer SQL-Ausdruck.
	 */
	@Deprecated
	public SQLExpression sql() {
		this.currentQuery = new SQLExpression();
		return this.currentQuery;
	}
	
	/**
	 * Setzt einen zurvor aufgebauten SELECT-Ausdruck an die Datenbank ab und liefert das Ergebnis zurueck.
	 * @return Das Ergebnis der SELECT-Anfrage.
	 * @throws DatabaseException
	 */
	@Deprecated
	public SQLTable get() throws DatabaseException {
		return this.query( this.getCurrentQuery().getSelectSQL(true), this.getCurrentQuery().getVars() );
	}
	
	/**
	 * Setzt einen zuvor aufgebauten INSERT-Ausdruck an die Datenbank ab und liefert die Anzahl der eingefuegten Zeilen als Ergebnis zurueck.
	 * @param table Tabelle, auf der der INSERT-Ausdruck ausgewertet werden soll.
	 * @return Anzahl der eingefuegten Datensaetze.
	 * @throws DatabaseException
	 */
	@Deprecated
	public int insert( String table ) throws DatabaseException {
		return this.dataManipulationQuery( this.getCurrentQuery().getInsertSQL(table, true), this.getCurrentQuery().getVars() );
	}
	
	/**
	 * Setzt einen zuvor aufgebauten INSERT-Ausdruck an die Datenbank ab und liefert die Anzahl der eingefuegten Zeilen als Ergebnis zurueck.
	 * @param table Tabelle, auf der der INSERT-Ausdruck ausgewertet werden soll.
	 * @return Anzahl der eingefuegten Datensaetze.
	 * @throws DatabaseException
	 */
	@Deprecated
	public int insert( TableSchema table ) throws DatabaseException {
		return this.dataManipulationQuery( this.getCurrentQuery().getInsertSQL(table.name, true), this.getCurrentQuery().getVars() );
	}
	
	/**
	 * Setzt einen zuvor aufgebauten DELETE-Ausdruck an die Datenbank ab und liefert die Anzahl der geloeschten Zeilen als Ergebnis zurueck.
	 * @param table Tabelle, auf der der DELETE-Ausdruck ausgewertet werden soll.
	 * @return Anzahl der geloeschten Datensaetze.
	 * @throws DatabaseException
	 */
	@Deprecated
	public int delete( String table ) throws DatabaseException {
		return this.dataManipulationQuery( this.getCurrentQuery().getDeleteSQL(table, true), this.getCurrentQuery().getVars() );
	}

	/**
	 * Fuehrt eine beliebigen SQL-Query aus, auf den keine Antwort erwartet wird.
	 * Wird typischerweise für spezielle SQL-Anfragen wie COMMIT genutzt.
	 * @param sql SQL-Satement.
	 * @throws DatabaseException
	 */
	public void rawQuery( String sql ) throws DatabaseException {
		try {
			Statement stmt = this.connection.createStatement();
			// SQL-Query ausfuehren
			stmt.execute( sql );
			stmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Error in the SQL expression: " + exception.toString(), exception );
		}
	}
	
	/**
	 * Sendet einen selbstgebauten SQL-Ausdruck an die Datenbank, der ein ResultSet als Ergebnis liefert.
	 * 
	 * @param sql SQL-Ausdruck, der ein ResultSet generiert wie z.B. SELECT.
	 * @return ResultSet auf die Anfrage.
	 * @throws DatabaseException
	 */
	public SQLTable query( String sql ) throws DatabaseException {
		return this.query(sql, new Vector<Object>() );
	}
	
	/**
	 * Sendet einen selbstgebauten SQL-Ausdruck an die Datenbank, der ein ResultSet als Ergebnis liefert.
	 * 
	 * @param sql SQL-Ausdruck, der ein ResultSet generiert wie z.B. SELECT.
	 * @param vars Bind-Variablen fuer WHERE-Ausdruck.
	 * @return ResultSet auf die Anfrage.
	 * @throws DatabaseException
	 */
	public SQLTable query( String sql, List<Object> vars ) throws DatabaseException {
		SQLTable result = null;
		
		logger.debug(sql);
		
		PreparedStatement pstmt = this.getStatement( sql, vars );

		// SQL-Query ausfuehren
		try {
			ResultSet rs = pstmt.executeQuery();
			result = new SQLTable( rs );
			pstmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Fehler in der aktuellen SQL-Anfrage.", exception );
		}
		
		// Query neu initialisieren
		this.sql();
		
		return result;
	}
	
	private PreparedStatement getStatement( String sql, List<Object> vars ) throws DatabaseException {
		// Bind-Variablen in SQL-Query initialisieren
		PreparedStatement pstmt = null;
		try {
			pstmt = this.connection.prepareStatement( sql );

			int i = 1;
			for ( Object obj : vars ) {
				pstmt.setObject( i, obj );
				i++;
			}
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Fehler beim Ersetzen der Bind-Variablen (die Fragezeichen) in der SQL-Anfrage.", exception );
		}
		
		return pstmt;
	}
	
	/**
	 * Sendet einen selbstgebauten datenveraendernden SQL-Ausdruck an die Datenbank.
	 *
	 * @param sql SQL-Ausdruck, z.B. INSERT, UPDATE oder DELETE:
	 * @return Die Zahl der veraenderten Datensaetze.
	 * @throws DatabaseException
	 */
	public int dataManipulationQuery( String sql ) throws DatabaseException {
		return this.dataManipulationQuery(sql, new Vector<Object>() );
	}
	
	/**
	 * Sendet einen selbstgebauten datenveraendernden SQL-Ausdruck an die Datenbank.
	 *
	 * @param sql SQL-Ausdruck, z.B. INSERT, UPDATE oder DELETE:
	 * @param vars Bind-Variablen fuer WHERE-Ausdruck.
	 * @return Die Zahl der veraenderten Datensaetze.
	 * @throws DatabaseException
	 */
	public int dataManipulationQuery( String sql, List<Object> vars ) throws DatabaseException {
		int result = -1;
		
		// Bind-Variablen in SQL-Query initialisieren
		PreparedStatement pstmt = null;
		try {
			pstmt = this.connection.prepareStatement( sql );

			int i = 1;
			for ( Object obj : vars ) {
				pstmt.setObject( i, obj );
				i++;
			}
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Error replacing bind variables (the questionmarks) in the current sql statement.", exception );
		}

		// SQL-Query ausfuehren
		try {
			result =  pstmt.executeUpdate();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Error in the current sql statement: "+exception.toString(), exception );
		}
		
		// Statement schliessen.
		try {
			pstmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Unable to close sql statement.", exception );
		}

		// Query neu initialisieren
		this.sql();
		
		return result;
	}
	
	/**
	 * Kann nach Auswertung eines INSERT-Ausdruck (siehe insert()) die ggf. automatisch generierten Spaltenwerte abrufen.
	 * Wurden keine Spaltenwerte generiert kann ein leeres ResultSet oder null zurueckgegeben werden.
	 * 
	 * Die Implementierung ist Datenbankherstellerabhaengig. Die Methode in SQLDatabase ist nur in leerer Dummy und sollte
	 * ueberschrieben werden.
	 * @return ResultSet, in dem die automatisch generierten Spalten enthalten sind. Kann auch leer oder null sein, wenn kein Wert generiert wurde.
	 */
	public SQLTable getGeneratedValues() {
		return null;
	}
	
	
	/**
	 * Gibt eine Liste mit allen in der aktuell ausgewaehlten Datenbank verfuegbaren Tabellen zurueck.
	 * @return Liste aller verfuegbarer Tabellen.
	 * @throws DatabaseException
	 */
	protected abstract List<String> getTableNames() throws DatabaseException;
	
	/**
	 * Gibt die Spaltennamen einer bestimmten Tabelle als Array zurueck.
	 * @param table Tabelle, von der die Spaltennamen zurueckgegeben werden sollen.
	 * @return Array mit allen Spaltennamen der ausgewaehlten Tabelle.
	 * @throws DatabaseException
	 */
	public String[] getColumnNames( String table ) throws DatabaseException {
		String[] result = null;
		
		this.sql().select("*").from(table).where("rownum", 1, Condition.LESSEQUAL);
		PreparedStatement pstmt = this.getStatement( this.getCurrentQuery().getSelectSQL(true), this.getCurrentQuery().getVars() );
		
		try {
			ResultSet rs = pstmt.executeQuery();
			
			int size = rs.getMetaData().getColumnCount();
			result = new String[size];
			for ( int i = 0; i < size; i++ ) {
				// Spalten fangen bei 1 an, deshalb i+1
				result[i] = rs.getMetaData().getColumnName(i+1);
			}
			pstmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Fehler beim Abfragen aller Spaltennamen.", exception );
		}
		
		return result;
	}
	
	/**
	 * Gibt eine Liste von Tupeln zurueck, die die Spaltennamen zusammen mit dem Typ einer bestimmten Tabelle
	 * enthalten.
	 * @param relation
	 * @return Liste von Tupeln. Die erste Tupelkomponente ist ein Attributname, die zweite Komponente ist der Attributtyp.
	 * @throws DatabaseException 
	 */
	public List<Tuple<String,String>> getColumnNamesWithAttributeTypes( String relation ) throws DatabaseException {
		
		List<Tuple<String,String>> attributes = new ArrayList<Tuple<String,String>>();
		
		// DESCRIBE liefert eine Tabelle mit einer Beschreibung einer Tabelle.
		// Sie enthaelt alle Namen und Typen von Atributen.
		this.sql().select("COLUMN_NAME").select("DATA_TYPE").select("DATA_LENGTH").from("USER_TAB_COLUMNS").where("TABLE_NAME", relation).order_by("column_id");
		SQLTable tableTypes = this.get();
		
		for ( String[] row : tableTypes ){
			attributes.add( new Tuple<String,String>(row[0],row[1]+"("+row[2]+")") );
		}
		
		return attributes;
	}
	
	/**
	 * Erstellt eine neue Tabelle mit dem Namen relationname und den Attributen attributes in der Datenbank.
	 * attributes ist eine Liste von Tupel<String,String>. Der erste Parameter ist dabei der Attributname,
	 * der zweite Parameter ist der Typ des Attributs. Der Typ muss dabei auch die Groesse enthalten (beispielsweise varchar(50)).
	 * @param relationname
	 * @param attributes
	 * @throws DatabaseException 
	 */
	public void createNewTable( String relationname, List<Tuple<String,String>> attributes ) throws DatabaseException{
		
		String sql = "CREATE TABLE " + relationname + " (";
		
		String sep = "";
		for ( Tuple<String,String> t : attributes ){
			sql += sep + t.getFirstElement() + " " + t.getSecondElement()+ " NOT NULL";
			sep = ", ";
		}
		
		sql += ")";
		
		this.dataManipulationQuery( sql );
	}
	
	/**
	 * Loescht die Tabelle relation aus der Datenbank. Dabei werden allerdings keine foreign-keys beruecksichtigt. 
	 * @param relation Name der Tabelle, die geloescht werden soll.
	 * @throws DatabaseException
	 */
	public void dropTable( String relation ) throws DatabaseException {
		this.dropTable(relation, false);
	}
	
	/**
	 * Loescht die Tabelle relation aus der Datenbank. Constraints werden ggf. kaskadierend geloescht, d.h.
	 * die Tabelle wird in jedem Fall geloescht.
	 * 
	 * @param relation Name der Tabelle, die geloescht werden soll.
	 * @param cascadeConstraints true, falls Constraints kaskadierend geloescht werden sollen, sonst false.
	 * @throws DatabaseException
	 */
	public void dropTable( String relation, boolean cascadeConstraints ) throws DatabaseException {
		String sql = "DROP TABLE \"" + relation + "\"";
		if ( cascadeConstraints ) {
			sql += " cascade constraints";
		}
		this.dataManipulationQuery( sql );
	}
	
	/**
	 * Loescht die Spalte column aus der Tabelle relation
	 * @param relationname
	 * @param columnname
	 * @throws DatabaseException
	 */
	public void dropColumn( String relation, String column ) throws DatabaseException {
		String sql = "ALTER TABLE " + relation + " DROP COLUMN " + column;
		this.dataManipulationQuery( sql );
	}
	
	/**
	 * Returns all values for a given relation (table) and attribute (column).
	 * @param relationname Relation or table name.
	 * @param attributename Attribute or column name.
	 * @return Set of all possible values for the attribute in the db (in that column).
	 * @throws DatabaseException
	 */
	public Set<String> getAttributeValues(String relationname, String attributename) throws DatabaseException {
		Set<String> attributeValues = new HashSet<String>();
		
		String sql = "SELECT DISTINCT "+attributename+" FROM "+relationname;
		
		SQLTable dbResult = this.query(sql);
		for ( String[] row : dbResult ) {
			attributeValues.add( row[0] );
		}
		
		return attributeValues;
	}
	
	public void call( String call ) throws DatabaseException {
		try {
			CallableStatement cstmt = this.connection.prepareCall( call );
			// Call ausfuehren
			cstmt.execute();
			cstmt.close();
		} catch ( SQLException exception ) {
			throw new DatabaseException( "Fehler in der aktuellen SQL-Anfrage.", exception );
		}
	}

	public void truncateTable(String relation) throws DatabaseException {
		String sql = "TRUNCATE TABLE " + relation;
		this.rawQuery(sql);
	}
	
	
	
	/**
	 * Reloads the available tables (relations) in the database.
	 * Should only be called if the database tables have changed.
	 * @throws DatabaseException
	 */
	public void reloadSchemaInformation() throws DatabaseException {
		this.dataCache = new HashMap<String, List<Column>>();

		for ( String relation : this.getTableNames() ) {
			List<Column> attributes = new LinkedList<Column>();
			for ( Tuple<String, String> attribute : this.getColumnNamesWithAttributeTypes(relation) ) {
				attributes.add( new Column(this, relation, attribute.getFirstElement(), attribute.getSecondElement()) );
			}
			
			this.dataCache.put(relation, attributes);
		}
	}
	
	/**
	 * Returns the names of all available tables (relations).
	 * @return Set of all table names in the database.
	 */
	public Set<String> getTables() {
		return this.dataCache.keySet();
	}
	
	/**
	 * Returns the columns (attributes) of a specific table (relation.
	 * Each column (attribute) consists of the corresponding column name
	 * and type and has a reference to the dictionaries (only if appdb).
	 * 
	 * @param name Name of the table (relation).
	 * @return List (the order is important) of columns (attributes).
	 */
	public List<Column> getColumns( String name ) {
		return this.dataCache.get(name.toUpperCase());
	}
}
