package server.database.schema.app;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class RefusalCensorExample extends MaintenanceDatabaseSchema {

	public final IllTableSchema ill;
	public final TreatTableSchema treat;
	
	public RefusalCensorExample() throws DatabaseException {
		super("Refusal Censor Example");
		
		this.ill = new IllTableSchema();
		this.treat = new TreatTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.ill );
		this.appTables.add( this.treat );
	}
	
	/**
	 *  db = { ill("lisa","flu"), ill("mary","aids"), ill("mike","flu"), ill("john","flu"),
	 *  		ill("kate","flu"), ill("james","flu"),
	 *			treat("mary","meda"), treat("mary","medb"), treat("lisa","medb") }
	 */
	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillIll(db);
		this.fillTreat(db);
	}
	
	private void fillIll(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Lisa" );
		sql.set( IllTableSchema.DIAGNOSIS, "Flu" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mary" );
		sql.set( IllTableSchema.DIAGNOSIS, "Aids" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Mike" );
		sql.set( IllTableSchema.DIAGNOSIS, "Flu" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "John" );
		sql.set( IllTableSchema.DIAGNOSIS, "Flu" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "Kate" );
		sql.set( IllTableSchema.DIAGNOSIS, "Flu" );
		db.insert(this.ill);
		
		sql = db.sql();
		sql.set( IllTableSchema.NAME, "James" );
		sql.set( IllTableSchema.DIAGNOSIS, "Flu" );
		db.insert(this.ill);
	}
	
	private void fillTreat(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineA" );
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "Mary" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineB" );
		db.insert(this.treat);
		
		sql = db.sql();
		sql.set( TreatTableSchema.NAME, "lisa" );
		sql.set( TreatTableSchema.PRESCRIPTION, "MedicineB" );
		db.insert(this.treat);
	}
	
	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		super.createDefaultAccessRights(db);
		int userId = super.createTestUser(db, 1, "PreCQE");
		try {
			this.fillPrior(db.getUserManagement().load(userId));
			this.fillConfidentialityPolicy(db.getUserManagement().load(userId));
		} catch (UnknownUserException e) {
			throw new DatabaseException("User does not exists, check your code.", e);
		}
	}
	
	public void fillPrior(User user) throws DatabaseException {
		try {
			PriorUser prior = user.getPriorUser();
			prior.add(new Formula("EXISTS B (treat(B,\"MedicineA\") AND ill(B,\"Flu\"))"));
			prior.add(new Formula("NOT EXISTS B (treat(B,\"MedicineA\") AND treat(B,\"MedicineB\") AND NOT ill(B,\"Aids\"))"));
			prior.add(new Formula("NOT EXISTS B (treat(B,\"MedicineA\") AND treat(B,\"MedicineB\") AND NOT ill(B,\"Aids\"))"));
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			policy.add(new Formula("EXISTS B ill(B,\"Aids\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
			policy.add( new Formula("ill(\"Lisa\",\"Flu\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
}
