package server.database.schema.app;

import communication.CqeType;

import exception.DatabaseException;
import exception.ParserException;
import exception.UnknownUserException;
import server.data.ConfidentialityPolicy;
import server.data.PriorUser;
import server.data.User;
import server.database.NewMaintenanceDatabase;
import server.database.schema.app.dict.DiagnosisDictionary;
import server.database.schema.app.dict.EffectsDictionary;
import server.database.schema.app.dict.NameDictionary;
import server.database.schema.app.dict.PrescriptionDictionary;
import server.database.schema.maintenance.MaintenanceDatabaseSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLExpression;
import server.parser.Formula;

public class PreCQEIllTreatEffectsExample extends MaintenanceDatabaseSchema {

	public final IllTableSchema ill;
	public final TreatTableSchema treat;
	public final EffectsTableSchema effects;
	
	public final NameDictionary nameDict;
	public final DiagnosisDictionary diagnosisDict;
	public final PrescriptionDictionary prescriptionDict;
	public final EffectsDictionary effectsDict;
	
	// Defines the used names for the example.
	private final String[] names = {"Mary", "Lisa", "Pete", "Theodor", "Kate", "Anne", "Liz", "Sam", "Abigail" , "Jonathan"};
	private final String[] letters = {"A", "B", "C", "D", "E", "F", "G", "H", "I", "J"};
	private final String[] numbers = {"0","1", "2","3","4","5","6","7","8","9"};
	
	private String[] manyNames = new String[names.length*letters.length*numbers.length];
	

	
	
	
	//private final String[] names = { "Mary", "Lisa", "Pete" };
	
	public PreCQEIllTreatEffectsExample() throws DatabaseException {
		super("PreCQE Ill, Treat, Effects Example");
		createManyNames();
		
		this.ill = new IllTableSchema();
		this.treat = new TreatTableSchema();
		this.effects = new EffectsTableSchema();
		
		// Add all tables to one list so that it can be iterated in the correct order easily.
		this.appTables.add( this.ill );
		this.appTables.add( this.treat );
		this.appTables.add( this.effects );
		
		this.nameDict = new NameDictionary();
		this.diagnosisDict = new DiagnosisDictionary();
		this.prescriptionDict = new PrescriptionDictionary();
		this.effectsDict = new EffectsDictionary();
		
		this.maintenanceTables.add( this.nameDict );
		this.maintenanceTables.add( this.diagnosisDict );
		this.maintenanceTables.add( this.prescriptionDict );
		this.maintenanceTables.add( this.effectsDict );
	}
	
	private void createManyNames() {
		String currentName;
		int count = 0;
		
		for (int i=0; i < names.length; i++) {
			for (int j=0; j < letters.length; j++) {
				for (int k=0; k < numbers.length; k++) {
					currentName = names[i].concat(letters[j].concat(numbers[k]));
					manyNames[count] = currentName;
					count++;
				}
			}
			
		}
	}

	@Override
	public void fillAppDBWithContents(SQLDatabase db) throws DatabaseException {
		this.fillIll(db);
		this.fillEffects(db);
		this.fillTreat(db);
	}
	
	public void fillIll(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		for ( String name : manyNames ) {
			sql = db.sql();
			sql.set( IllTableSchema.NAME, name );
			sql.set( IllTableSchema.DIAGNOSIS, "Aids" );
			db.insert(this.ill);
		}
	}
	
	public void fillEffects(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		for ( String name : manyNames ) {
			sql = db.sql();
			sql.set( EffectsTableSchema.NAME, name );
			sql.set( EffectsTableSchema.EFFECT, "HighCosts" );
			db.insert(this.effects);
		}
	}
	
	public void fillTreat(SQLDatabase db) throws DatabaseException {
		SQLExpression sql;
		
		for ( String name : manyNames ) {
			sql = db.sql();
			sql.set( TreatTableSchema.NAME, name );
			sql.set( TreatTableSchema.PRESCRIPTION, "MedicineA" );
			db.insert(this.treat);
			sql = db.sql();
			sql.set( TreatTableSchema.NAME, name );
			sql.set( TreatTableSchema.PRESCRIPTION, "MedicineB" );
			db.insert(this.treat);
		}
	}	

	@Override
	public void fillMaintenanceDBWithContents(NewMaintenanceDatabase db) throws DatabaseException {
		super.createDefaultAccessRights(db);
		super.createAdminUser(db);
		int userId = super.createTestUser(db, 1, "PreCQE");
		try {
			this.fillPrior(db.getUserManagement().load(userId));
			this.fillConfidentialityPolicy(db.getUserManagement().load(userId));
		} catch (UnknownUserException e) {
			throw new DatabaseException("User 'test' does not exists, check your code.", e);
		}
		
		super.linkDictionary( db, this.nameDict, IllTableSchema.NAME );
		super.linkDictionary( db, this.diagnosisDict, IllTableSchema.DIAGNOSIS );
		super.linkDictionary( db, this.nameDict, TreatTableSchema.NAME );
		super.linkDictionary( db, this.prescriptionDict, TreatTableSchema.PRESCRIPTION );
		super.linkDictionary( db, this.nameDict, EffectsTableSchema.NAME );
		super.linkDictionary( db, this.effectsDict, EffectsTableSchema.EFFECT );
	}
	
	public void fillPrior(User user) throws DatabaseException {
		try {
			PriorUser prior = user.getPriorUser();
			prior.add(new Formula("EXISTS X (treat(X,\"MedicineB\") AND ill(X,\"Flu\"))"));
			prior.add(new Formula("FORALL X ((treat(X,\"MedicineA\") AND treat(X,\"MedicineB\")) IMPL ill(X,\"Aids\"))"));
			prior.add(new Formula("FORALL X (ill(X,\"Aids\") IMPL effects(X,\"HighCosts\"))"));
			prior.add(new Formula("EXISTS X (ill(X,\"Aids\"))"));
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
	
	public void fillConfidentialityPolicy(User user) throws DatabaseException {
		try {
			ConfidentialityPolicy policy = user.getConfidentialityPolicy();
			for ( String name : manyNames ) {
				policy.add( new Formula("effects(\""+name+"\",\"HighCosts\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
//				policy.add( new Formula("effects(\"Abigail1A\",\"HighCosts\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);
//				policy.add(new Formula("effects(\"MaddinGibbetNich\",\"HighCosts\")"), CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS);

			}
		} catch ( ParserException e ) {
			throw new DatabaseException("Formula in example wrong, check your code.", e);
		}
	}
}
