package server.database;

import java.util.ArrayList;
import java.util.List;

import server.database.sql.SQLDatabase;
import server.parser.Formula;
import server.parser.node.Node;
import exception.DatabaseException;
import exception.UnsupportedFormulaException;

/**
 * Ermoeglicht den Zugriff auf die eigentliche Nutzdatenbank.
 * 
 * Bei einer unvollstaendigen Datenbank existiert zu jeder Relation zwei Tabellen:
 * eine fuer nicht negierte Fakten und einmal fuer negierte Fakten (Tabellenname setzt sich aus dem Prefix
 * "not_" und dem Namen der Relation zusammen). Der Administrator muss darauf achten, dass negative Informationen
 * explizit in der entsprechenden "not_" Tabelle eingetragen sind.
 */
public abstract class AppDatabase {

	public final static int EVAL_FALSE = 0;
	public final static int EVAL_TRUE = 1;
	public final static int EVAL_UNDEF = -1;
	
	/**
	 * Wertet eine gegebene Formel gegen eine vollstaendige Datenbankinstanz aus. Gibt true zurueck, wenn die Datenbankinstanz
	 * die Formel erfüllt (I^db |= Phi), ansonsten false. Die Methode entspricht auch der in der Literatur mit eval bezeichneten
	 * Funktion.
	 * 
	 * @param interaction Fuer diese Formel wird geprueft, ob die Datenbankinstanz sie erfuellt.
	 * @return true, falls I^db |= Phi (fuer I^db = aktuelle Datenbankinstanz und Phi = Formel) gilt, sonsten false
	 * @throws DatabaseException Kann bei Problemen mit der Datenbankverbindung auftreten.
	 * @throws UnsupportedFormulaException Kann bei der Umwandlung der Formel in eine SQL-Anfrage auftreten.
	 */
	public abstract boolean evalComplete( Formula interaction ) throws DatabaseException, UnsupportedFormulaException;
	
	/**
	 * Nutzt evalComplete(), um den Wahrheitswert der Anfrage in der Datenbank zu ueberpruefen.
	 * Negiert die Anfrage, falls die Anfrage in der Datenbank nicht wahr ist. Ansonsten wird
	 * die Anfrage unveraendert zurueckgegeben.
	 * 
	 * @param interaction Fuer diese Formel wird geprueft, ob die Datenbankinstanz sie erfuellt. Wird ggf. negiert zurueckgegeben.
	 * @return Die Interaktion, falls evalComplete() true ergeben hat, ansonsten die Negation der Interaktion.
	 * @throws DatabaseException Kann bei Problemen mit der Datenbankverbindung auftreten.
	 * @throws UnsupportedFormulaException Kann bei der Umwandlung der Formel in eine SQL-Anfrage auftreten.
	 * @see #evalComplete(Node)
	 */
	public Formula evalStarComplete( Formula interaction ) throws DatabaseException, UnsupportedFormulaException {
		if( !this.evalComplete( interaction ) ) {
			interaction.negate();
		}
		
		return interaction;
	}
	
	/**
	 * Wertet eine gegebene Formel gegen eine unvollstaendige Datenbankinstanz aus. Gibt EVAL_TRUE zurueck, wenn alle Modelle
	 * der Datenbankinstanz auch Modell der Formel sind (db |= Phi). Gibt EVAL_FALSE zurueck, wenn alle Modelle der Datenbankinstanz
	 * auch Modelle der Negation der Formel sind (db |= -Phi). Treffen beide Faelle nicht zu, wird EVAL_UNDEF zurueckgegeben.
	 * @param interaction Diese Formel wird gegen die unvsollstaendige Datenbankinstanz ausgewertet.
	 * @return EVAL_TRUE, falls db |= Phi, (db = Menge von Saetzen, Phi = Formel), EVAL_FALSE falls db |= -Phi (- = Negation), sonst EVAL_UNDEF
	 */
	public abstract int evalIncomplete( Formula interaction ) throws DatabaseException;
	
	/**
	 * Wertet eine offene Anfrage aus. Als Ergebnis wird eine Liste mit allen Instanzen der
	 * Anfrage zurueckgegeben, die in der Datenbank wahr sind.
	 * 
	 * @param interaction
	 * @return
	 * @throws DatabaseException
	 * @throws UnsupportedFormulaException
	 */
	public abstract List<Formula> evalPosComplete( Formula interaction ) throws DatabaseException, UnsupportedFormulaException;
	
	/**
	 * Aktualisiert die Datenbank und aendert den Wert des uebergebenen Literals. Ist das Literal negiert, so
	 * wird das dazugehoerige Tupel (falls vorhanden) aus der Datenbank geloescht. Ist das Literal nicht negiert,
	 * so wird ein neues Tupel mit dem Inhalt des Literals eingefuegt.
	 * 
	 * @param literal Literal, das danach in der Datenbank (nicht) gelten sollen.
	 * @throws DatabaseException Kann bei Problemen mit der Datenbankverbindung oder bei der Erfuellung von Constraints auftreten.
	 * @throws UnsupportedFormulaException Wird erzeugt, wenn die uebergebene Node kein Literal ist.
	 */
	public void updateComplete( Formula literal ) throws DatabaseException, UnsupportedFormulaException {
		List<Formula> literals = new ArrayList<Formula>();
		literals.add( literal );
		this.updateComplete( literals );
	}
	
	/**
	 * Aktualisiert die Datenbank und aendert die Werte der uebergebenen Literale. Ist ein Literal negiert, so
	 * wird das dazugehoerige Tupel (falls vorhanden) aus der Datenbank geloescht. Ist ein Literal nicht negiert,
	 * so wird ein neues Tupel mit dem Inhalt des Literals eingefuegt.
	 * 
	 * @param literals Liste mit Literalen, die danach in der Datenbank (nicht) gelten sollen.
	 * @throws DatabaseException Kann bei Problemen mit der Datenbankverbindung oder bei der Erfuellung von Constraints auftreten.
	 * @throws UnsupportedFormulaException Wird erzeugt, wenn die uebergebene Liste nicht nur aus Literalen besteht.
	 * @return The number of tuples updated (deleted or inserted).
	 */
	public abstract int updateComplete( List<Formula> literals ) throws DatabaseException, UnsupportedFormulaException;
	
	/**
	 * Gibt ein Array der Attributnamen einer Relation zurück.
	 * 
	 * @param relationname Name der Relation, dessen Attributnamen zurückgegeben werden sollen
	 * @return Array von Attributnamen der Relation relationname
	 */
	public abstract String[] getAttributeNames(String relationname) throws DatabaseException;
	
	public abstract void truncateRelation(String relationname) throws DatabaseException;
	
	/**
	 * Returns the currently used sql database.
	 * @return SQL-Database instance.
	 */
	public abstract SQLDatabase getSQLDatabase();
}
