package server.data;

import java.io.Serializable;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;

import server.database.schema.Schema;
import server.database.schema.maintenance.RightTableSchema;
import server.database.schema.maintenance.RoleRightTableSchema;
import server.database.schema.maintenance.RoleTableSchema;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;

import communication.CqeType;
import communication.CqeType.Right;
import exception.DatabaseException;

/**
 * A role is a collection of rights.
 * It's not possible to directly extend EnumSet, which is why this class
 * extends AbstractSet instead and delegates all calls to EnumSet. The
 * delegation code was auto-generated.
 * 
 * FIXME: port to new data abstraction layer
 */
public class Role extends AbstractSet<Right> implements Serializable, Cloneable {
	private static final long serialVersionUID = -5363448735266888578L;
	
	private EnumSet<Right> rights;
	private String name;
	private int roleId;
	
	public static Role load(SQLDatabase db, int roleId) throws DatabaseException {
		Collection<Role> roles = Role.loadRole(db, roleId);
		
		return roles.iterator().next();
	}
	
	public static Collection<Role> loadAll( SQLDatabase db ) throws DatabaseException {
		return Role.loadRole(db, -1);
	}
	
	private static Collection<Role> loadRole( SQLDatabase db, int roleId ) throws DatabaseException {
		HashMap<Integer,Role> roles = new HashMap<Integer,Role>();

		SQLSelect sql = db.newSelect();
		sql.select(RoleTableSchema.ID).select(RightTableSchema.NAME).select(RoleTableSchema.NAME);
		sql.from(Schema.maintenanceDatabase.role);
		sql.join(Schema.maintenanceDatabase.role_right, RoleTableSchema.ID, RoleRightTableSchema.ROLE_ID);
		sql.join(Schema.maintenanceDatabase.right, RoleRightTableSchema.RIGHT_ID, RightTableSchema.ID);
		// Only select one role if needed.
		if ( roleId != -1 ) {
			sql.where(RoleTableSchema.ID, roleId);
		}
		
		SQLTable table = sql.get();

		for ( String[] row : table ) {
			int id = Integer.parseInt( row[0] );
			String right = row[1];
			String role = row[2];
			
			Role currentRole = null;
			if ( roles.containsKey(id) ) {
				currentRole = roles.get(id);
			} else {
				currentRole = new Role(id, role);
				roles.put(id, currentRole);
			}
			
			currentRole.add( right );
		}

		return roles.values();
	}
	
	private Role( int roleId, String name ) {
		this.rights = EnumSet.noneOf(Right.class);
		this.roleId = roleId;
		this.name = name;
	}
	
	public String getName( ) {
		return this.name;
	}
	
	public EnumSet<Right> getRights() {
		return this.rights.clone();
	}
	
	public int getId() {
		return this.roleId;
	}
	
	/**
	 * Fuegt ein Recht mit dem angegebenen Namen hinzu. Die Namen der Rechte entsprechen den Konstantenbezeichnern in der Klasse CqeType.Right.
	 * @param right Name des Rechts.
	 * @return True, falls das Recht hinzugefuegt werden konnte, sonst false.
	 */
	public boolean add( String right ) {
		return this.add( CqeType.Right.stringToRight(right) );
	}

	/**
	 * @param e
	 * @return
	 * @see java.util.AbstractCollection#add(java.lang.Object)
	 */
	public boolean add(Right e) {
		return rights.add(e);
	}

	/**
	 * @param c
	 * @return
	 * @see java.util.AbstractCollection#addAll(java.util.Collection)
	 */
	public boolean addAll(Collection<? extends Right> c) {
		return rights.addAll(c);
	}

	/**
	 * 
	 * @see java.util.AbstractCollection#clear()
	 */
	public void clear() {
		rights.clear();
	}

	/**
	 * @return
	 * @see java.util.EnumSet#clone()
	 */
	public EnumSet<Right> clone() {
		return rights.clone();
	}

	/**
	 * @param o
	 * @return
	 * @see java.util.AbstractCollection#contains(java.lang.Object)
	 */
	public boolean contains(Object o) {
		return rights.contains(o);
	}

	/**
	 * @param c
	 * @return
	 * @see java.util.AbstractCollection#containsAll(java.util.Collection)
	 */
	public boolean containsAll(Collection<?> c) {
		return rights.containsAll(c);
	}

	/**
	 * @param o
	 * @return
	 * @see java.util.AbstractSet#equals(java.lang.Object)
	 */
	public boolean equals(Object o) {
		
		if (o instanceof Role) {
			Role e = (Role)o;
			return e.getId() == roleId;
		}
		
		return false;
		
	}

	/**
	 * @return
	 * @see java.util.AbstractSet#hashCode()
	 */
	public int hashCode() {
		return rights.hashCode();
	}

	/**
	 * @return
	 * @see java.util.AbstractCollection#isEmpty()
	 */
	public boolean isEmpty() {
		return rights.isEmpty();
	}

	/**
	 * @return
	 * @see java.util.AbstractCollection#iterator()
	 */
	public Iterator<Right> iterator() {
		return rights.iterator();
	}

	/**
	 * @param o
	 * @return
	 * @see java.util.AbstractCollection#remove(java.lang.Object)
	 */
	public boolean remove(Object o) {
		return rights.remove(o);
	}

	/**
	 * @param c
	 * @return
	 * @see java.util.AbstractSet#removeAll(java.util.Collection)
	 */
	public boolean removeAll(Collection<?> c) {
		return rights.removeAll(c);
	}

	/**
	 * @param c
	 * @return
	 * @see java.util.AbstractCollection#retainAll(java.util.Collection)
	 */
	public boolean retainAll(Collection<?> c) {
		return rights.retainAll(c);
	}

	/**
	 * @return
	 * @see java.util.AbstractCollection#size()
	 */
	public int size() {
		return rights.size();
	}

	/**
	 * @return
	 * @see java.util.AbstractCollection#toArray()
	 */
	public Object[] toArray() {
		return rights.toArray();
	}

	/**
	 * @param <T>
	 * @param a
	 * @return
	 * @see java.util.AbstractCollection#toArray(T[])
	 */
	public <T> T[] toArray(T[] a) {
		return rights.toArray(a);
	}

	public String toString() {
		return this.getName();
	}
	
}
