package server.core.notificationHandler;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import org.apache.log4j.Logger;

import server.core.Client;

import notification.Notification;

/**
 * Handles all incoming notifications and distributes the processing
 * to registered handlers. The handlers must extend the NotificationHandler
 * class with the generic type of the notification which they can process.
 * There can be more than one notification handler per notification, but no
 * execution order is guaranteed. In fact the notification will likely be
 * handled concurrent, so be aware of synchronization.
 */
public class NotificationHub {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.core");
	
	private ExecutorService pool = null;
	private Map<Class<? extends Notification>, Set<NotificationHandler<? extends Notification>>> handlers = null; 
	
	/**
	 * Creates a NotificationHub with all essential/core notification handlers already registered.
	 * @return
	 */
	public static NotificationHub createBaseHub() {
		NotificationHub hub = new NotificationHub();
		
		// Login & Interactions
		hub.register( new LoginNotificationHandler() );
		hub.register( new InteractionNotificationHandler() );
		
		// User & Role Management
		hub.register( new GetAllUsersNotificationHandler() );
		hub.register( new GetAllRolesNotificationHandler() );
		hub.register( new AddUserNotificationHandler() );
		hub.register( new EditUserNotificationHandler() );
		hub.register( new DeleteUserNotificationHandler() );
		hub.register( new AddUserInformationNotificationHandler() );
		hub.register( new EditUserInformationNotificationHandler() );
		hub.register( new DeleteUserInformationNotificationHandler() );
		
		return hub;
	}
	
	/**
	 * Creates a new NotificationHub with no registered handlers.
	 */
	public NotificationHub() {
		// Create a pool of 20 threads ready to handle all incoming notifications.
		this.pool = Executors.newFixedThreadPool( 20, new ThreadFactory() {
			private int threadCounter = 0;
			
			@Override
			public Thread newThread(Runnable arg0) {
				Thread thread = new Thread(arg0, "Handler Pool - Thread #"+this.threadCounter);
				this.threadCounter++;
				return thread;
			}
		});
		
		// Create map for all handlers.
		this.handlers = new HashMap<Class<? extends Notification>, Set<NotificationHandler<? extends Notification>>>();
	}
	
	/**
	 * Registers a new NotificationHandler to the hub.
	 * The notification type must also be set in the constructor of the
	 * notification handler (e.g. "super(LoginNotification.class);").
	 * @param handler
	 */
	public synchronized void register( NotificationHandler<? extends Notification> handler ) {
		if ( handler == null ) {
			throw new IllegalArgumentException();
		}
		
		Class<? extends Notification>notificationClass = handler.getHandledClass();
		Set<NotificationHandler<? extends Notification>> handlerSet = null;
		
		if ( this.handlers.containsKey(notificationClass) ) {
			handlerSet = this.handlers.get( notificationClass );
		} else {
			handlerSet = new HashSet<NotificationHandler<? extends Notification>>();
			this.handlers.put( notificationClass, handlerSet );
		}
		
		handlerSet.add( handler ); 
	}
	
	/**
	 * Unregisters a new NotificationHandler from the hub.
	 * The notification type must also be set in the constructor of the
	 * notification handler (e.g. "super(LoginNotification.class);").
	 * @param handler
	 */
	public synchronized void unregister( NotificationHandler<? extends Notification> handler ) {
		if ( handler == null ) {
			throw new IllegalArgumentException();
		}
		
		Class<? extends Notification>notificationClass = handler.getHandledClass();
		
		if ( this.handlers.containsKey(notificationClass) ) {
			Set<NotificationHandler<? extends Notification>> handlerSet = this.handlers.get( notificationClass );
			handlerSet.remove( handler );
		}
	}
	
	/**
	 * This method informs the hub about a new notification received from a client.
	 * It distributes the handling of the notification to the notification handlers,
	 * which run in separate threads.
	 * 
	 * This method should only be called by the Server.
	 * 
	 * @param client
	 * @param notification
	 */
	public synchronized void handle( Client client, Notification notification ) {
		logger.debug("Received notification "+notification+" from client "+client);
		if ( this.handlers.containsKey(notification.getClass()) ) {
			Set<NotificationHandler<? extends Notification>> handlerSet = this.handlers.get( notification.getClass() );
			
			for ( NotificationHandler<? extends Notification> handler : handlerSet ) {
				handler.setInput( client, notification );
				this.pool.execute( handler );
			}
		}
	}
}
