package server.core.notificationHandler;

import java.io.IOException;
import java.util.LinkedList;

import communication.CqeType;

import exception.DatabaseException;
import exception.UnknownUserException;
import notification.clientToServer.LoginNotification;
import notification.serverToClient.ConfidentialityPolicyNotification;
import notification.serverToClient.ConstraintsNotification;
import notification.serverToClient.ExceptionNotification;
import notification.serverToClient.LogNotification;
import notification.serverToClient.LoginAppNotification;
import notification.serverToClient.PriorAllNotification;
import notification.serverToClient.PriorUserNotification;
import notification.serverToClient.StateNotification;
import server.core.Client;
import server.data.User;
import server.database.DatabaseConfiguration;
import server.database.DatabaseConfigurationList;
import server.database.NewMaintenanceDatabase;
import server.database.OracleSQLAppDatabase;
import server.util.Tuple;
import user.IdAndFormula;

public class LoginNotificationHandler extends NotificationHandler<LoginNotification> {

	public LoginNotificationHandler() {
		super( LoginNotification.class );
	}

	@Override
	protected void handle(Client client, LoginNotification notification) throws IOException, DatabaseException {
		// Get database connection for the selected database instance.
		for ( Tuple<DatabaseConfiguration,DatabaseConfiguration> config : DatabaseConfigurationList.DEFAULT_LIST ) {
			// FIXME: was ist mit dbms == XML?
			if ( config.getFirstElement().getId().equals(notification.getDbms()) ) {
				// Found the selected configuration.
				try {
					client.setMaintenanceDB( NewMaintenanceDatabase.load(config.getFirstElement()) );
					client.setApplicationDB( new OracleSQLAppDatabase(config.getSecondElement()) );
				} catch ( DatabaseException exception ) {
					client.send( new ExceptionNotification(exception, ExceptionNotification.CRITICAL) );
					return;
				}

				break;
			}
		}
		
		// Check if we got a database connection. If not, abort.
		if ( client.getMaintenanceDB() == null || client.getApplicationDB() == null ) {
			throw new DatabaseException("Unknown database instance selected. Aborting login.", null);
		}

		// Load user information.
		User user = null;
		try {
			user = client.getMaintenanceDB().getUserManagement().load( notification.getUsername() );
		} catch ( DatabaseException exception ) {
			client.send( new ExceptionNotification(exception, ExceptionNotification.CRITICAL) );
			return;
		} catch (UnknownUserException e) {
			client.send( new ExceptionNotification(e, ExceptionNotification.CRITICAL) );
			return;
		}

		// Check if passwords match.
		LoginAppNotification answer = null;

		// FIXME: This type of password comparison is prone to timing attacks.
		// The java guidelines also say storing a password (in general) is bad.
		// They suggest using char[] instead of string because the password
		// can be cleared out directly after using it (thus avoiding to wait
		// for garbage collection). Maybe we shouldn't store the password at
		// all and just let the database compare it.
		if ( user.getPassword().compareTo(notification.getPassword()) == 0 ) {
			answer = new LoginAppNotification( "Login successful", user.getRole().getRights() );
		} else {
			answer = new LoginAppNotification( "Username or password wrong" );
		}

		client.send( answer );

		if ( answer.getSuccessful() ) {
			// Store User in Client
			client.setUser( user );
			
			// Send all loaded User information to the client.
			client.send( new LogNotification(user.getLog().getCopyAsList()) );
				
			// Only send confidentiality poliy if the user has the right VIEW_CONFIDENTIALITY_POLICY.
			if ( user.getRole().contains(CqeType.Right.VIEW_CONFIDENTIALITY_POLICY) ) {
				client.send( new ConfidentialityPolicyNotification(user.getConfidentialityPolicy().getCopyAsList()) );
			} else {
				client.send( new ConfidentialityPolicyNotification(null) );
			}

			client.send( new ConstraintsNotification(new LinkedList<IdAndFormula>()) );
			client.send( new PriorAllNotification(new LinkedList<IdAndFormula>()) );
			client.send( new PriorUserNotification(user.getPriorUser().getCopyAsList()) );
			client.send( new StateNotification("CURRENTLY NOT WORKING DUE TO BACKEND CHANGES") );
		}
	}

}
