package server.censor.signature.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import server.database.AppDatabase;
import server.database.sql.SQLDatabase;
import server.database.sql.SQLTable;
import server.util.Tuple;
import exception.DatabaseException;


public class TemplateDependency {

	private String relationname;
	private int attributeCount;
	private List<TemplateDependencyRow> premises;
	private TemplateDependencyRow conclusion;
	
	
	public TemplateDependency(String relationname, int attributeCount) {
		this.relationname = relationname;
		this.attributeCount = attributeCount;
		this.premises = new ArrayList<TemplateDependencyRow>();
		this.conclusion = new TemplateDependencyRow(attributeCount);
		for( int i=0; i<attributeCount; ++i ) {
			this.conclusion.setElement(i, new DistinguishedSymbolElement("ds_"+i));
		}
	}
	
	public TemplateDependency(TemplateDependency copy) {
		this.relationname = copy.relationname;
		this.attributeCount = copy.attributeCount;
		this.premises = new ArrayList<TemplateDependencyRow>();
		for( TemplateDependencyRow premise : copy.premises ) {
			this.premises.add(new TemplateDependencyRow(premise));
		}
		this.conclusion = new TemplateDependencyRow(copy.conclusion);
	}
	
	public String getRelationname() {
		return this.relationname;
	}
	
	public void setRelationname(String relationname) {
		this.relationname = relationname;
	}
	
	public int getAttributeCount() {
		return this.attributeCount;
	}
	
	public int getPremiseCount() {
		return this.premises.size();
	}
	
	public TemplateDependencyRow getPremise(int i) {
		if( i < 0 || i >= this.premises.size() )
			return null;
		
		return premises.get(i);
	}
	
	public void setPremise(int i, TemplateDependencyRow premise) {
		if( i < 0 || i >= this.premises.size() || premise.getAttributeCount() != this.attributeCount )
			return;
		
		this.premises.set(i, premise);
	}
	
	public void addPremise(TemplateDependencyRow premise) {
		if( premise.getAttributeCount() != this.attributeCount )
			return;
		
		this.premises.add(premise);
	}
	
	public void removePremise(int i) {
		this.premises.remove(i);
	}
	
	public TemplateDependencyRow getConclusion() {
		return this.conclusion;
	}
	
	public void setConclusion(TemplateDependencyRow conclusion) {
		if( conclusion.getAttributeCount() != this.attributeCount )
			return;
		
		this.conclusion = conclusion;
	}
	
	
	public Set<Integer> getAttributesContainingDistinguishedSymbols() {
		Set<Integer> attributes = new HashSet<Integer>();
		for( int i=0; i<attributeCount; ++i ) {
			if( this.conclusion.getElement(i).isDistinguishedSymbol() ) {
				attributes.add( i );
			}
		}
		
		return attributes;
	}
	
	/**
	 * Gibt eine Map zurueck, die zu jedem ausgezeichnetem Symbol alle Positionen enthaelt an denen das ausgezeichnete Symbol vorkommt.
	 * Der Key der Map ist das Symbol selbst (value) und die Positionen sind als Tupel notiert: erste Element ist das Attribut, zweites
	 * Element die Praemissenzeile (-1 fuer die Konklusion).
	 *  
	 * @return
	 */
	public Map<String, Tuple<Integer, Set<Integer>>> getDistinguishedSymbolPositions() {
		Map<String, Tuple<Integer, Set<Integer>>> distinguishedSymbolPositons = new HashMap<String, Tuple<Integer, Set<Integer>>>();
		// ausgezeichnete Variable in der Konklusion suchen
		for( int i=0; i<attributeCount; ++i ) {
			if( conclusion.getElement(i).isDistinguishedSymbol() ) {
				distinguishedSymbolPositons.put(conclusion.getElement(i).getValue(), new Tuple<Integer, Set<Integer>>(i, new HashSet<Integer>()));
				distinguishedSymbolPositons.get(conclusion.getElement(i).getValue()).getSecondElement().add(-1);
				// Position der ausgezeichneten Variable in den Praemissen bestimmen
				for( int rowIndex=0; rowIndex<premises.size(); ++rowIndex ) {
					TemplateDependencyRow premise = premises.get(rowIndex);
					if( premise.getElement(i).isDistinguishedSymbol() ) {
						distinguishedSymbolPositons.get(premise.getElement(i).getValue()).getSecondElement().add(rowIndex);
					}
				}
			}
		}
		return distinguishedSymbolPositons;
	}
	
	public Map<String, Tuple<Integer, Set<Integer>>> getNonDistinguishedSymbolPositions() {
		Map<String, Tuple<Integer, Set<Integer>>> nonDistinguishedSymbolPositons = new HashMap<String, Tuple<Integer, Set<Integer>>>();
		// Alle nicht ausgezeichneten Variablen mit deren Positionen in die Liste aufnehmen
		for( int i=0; i<attributeCount; ++i ) {
			for( int rowIndex=0; rowIndex<premises.size(); ++rowIndex ) {
				TemplateDependencyRow premise = premises.get(rowIndex);
				if( premise.getElement(i).isNonDistinguishedSymbol() ) {
					if( ! nonDistinguishedSymbolPositons.containsKey(premise.getElement(i).getValue()) ) {
						nonDistinguishedSymbolPositons.put(premise.getElement(i).getValue(), new Tuple<Integer, Set<Integer>>(i, new HashSet<Integer>()));
					}
					nonDistinguishedSymbolPositons.get(premise.getElement(i).getValue()).getSecondElement().add(rowIndex);
				}
			}
		}
		return nonDistinguishedSymbolPositons;
	}
	
	/**
	 * Gibt eine Map zurueck, die zu jedem mehrfach vorkommenden nicht ausgezeichnetem Symbol (zu dem also eine Gleichheitsbedingung existiert) 
	 * alle Positionen (x=Zeilenindex, y=Spaltenindex) enthaelt an denen das nicht ausgezeichnete Symbol vorkommt.
	 * 
	 * @return
	 */
	public Map<String, Tuple<Integer, Set<Integer>>> getMultipleOccurringNonDistinguishedSymbolPositions() {
		Map<String, Tuple<Integer, Set<Integer>>> nonDistinguishedSymbolPositions = this.getNonDistinguishedSymbolPositions();
		Map<String, Tuple<Integer, Set<Integer>>> multipleOccurringNonDistinguishedSymbolPositions = new HashMap<String, Tuple<Integer, Set<Integer>>>();
		// Alle nicht ausgezeichneten Variablen mit mehr als einer Positionen übernehmen
		for( Map.Entry<String, Tuple<Integer, Set<Integer>>> symbolPositions : nonDistinguishedSymbolPositions.entrySet() ) {
			if( symbolPositions.getValue().getSecondElement().size() >= 1 ) {
				multipleOccurringNonDistinguishedSymbolPositions.put(symbolPositions.getKey(), symbolPositions.getValue());
			}
		}
		return multipleOccurringNonDistinguishedSymbolPositions;
	}
	
	public boolean nonDistinguishedSymbolOccursOnlyOnce(NonDistinguishedSymbolElement element) {
		return getNonDistinguishedSymbolPositions().get(element.getValue()).getSecondElement().size() == 1;
	}
	
	@Override
	public String toString() {
		String result = new String();
		
		// Relationenname
		result += "Relation: " + relationname + "\n";
		
		// Attributnamen
		for( int i=0; i<attributeCount; ++i ) {
			result += "attribute " + i + "\t";
		}
		result += "\n----------------------------------------\n";
		// Praemissen
		for( TemplateDependencyRow premise : premises ) {
			for( int i=0; i<attributeCount; ++i ) {
				result += premise.getElement(i).getValue() + "\t";
			}
			result += "\n";
		}
		result += "----------------------------------------\n";
		// Konklusion
		for( int i=0; i<attributeCount; ++i ) {
			result += conclusion.getElement(i).getValue() + "\t";
		}
		
		return result;
	}
	
	
	/**
	 * Diese Methode implementiert die Schritte 4 und 5 des statischen Ansatzes (vgl. S. 56 aus der Diplomarbeit von Torsten Schlotmann).
	 * 
	 * @param relationname	Relationenname der betrachteten Relation. Nur Template Dependencys mit diesem Relationennamen werden verarbeitet
	 * @param attributeNames Attributnamen der betrachteten Relation. Nur Template Dependencys mit uebereinstimmenden Attributnamen werden verarbeitet
	 * @param SigmaPlusMin	Kandidaten mit minimaler Prämisse und maximaler Konklusion, die von den semantischen Bedingungen Sigma impliziert werden
	 * @param pot_sec 		Menge der potentiellen Geheimnisse, wobei nur jene verwendet werden, dessen Relationenname und Attributnamen mit den ersten beiden Parametern uebereinstimmen
	 * @return				TD[phi]: Menge der Template Dependencies aus SigmaPlusMin, die mit den potentiellen Geheimnissen instantiiert wurden
	 */
	public static List<TemplateDependency> generateTDPhi(String relationname, List<TemplateDependency> SigmaPlusMin, List<SignatureConfidentialityPolicyItem> pot_sec) {
		// Schritt 4 und 5:
		// bestimme alle Paare (phi, TD) für die gilt, dass für alle Attribute, die in phi Konstanten enthalten, 
		// die Template Dependency TD eine ausgezeichnete Variable besitzt und Instantiiere die Template
		// Dependency mit phi
		List<TemplateDependency> TDPhis = new LinkedList<TemplateDependency>();
		
		// durchlaufe alle Kombinationen und pruefe ob Bedingung erfuellt ist
		for( SignatureConfidentialityPolicyItem phi : pot_sec) {
			if( ! phi.getRelationname().equals(relationname) ) {
				// potentielles Geheimnis nicht relevant, da es sich auf eine andere Relation bezieht
				continue;
			}
			
			for( TemplateDependency td : SigmaPlusMin ) {
				if( ! td.getRelationname().equals(relationname) ) {
					// Template Dependency nicht relevant, da es sich auf eine andere Relation bezieht
					continue;
				}
				
				// bestimme alle Attribute von phi, die mit Konstanten belegt sind
				Set<Integer> attributesWithConstants = new HashSet<Integer>();
				for( int i=0; i<td.getAttributeCount(); ++i ) {
					if( ! phi.getAttributeValue(i).equals("*") ) {
						attributesWithConstants.add(i);
					}
				}
				
				// pruefe ob alle Attribute, die in phi mit Konstanten belegt sind in td eine ausgezeichnete Variable besitzen
				if( td.getAttributesContainingDistinguishedSymbols().containsAll(attributesWithConstants) ) {
					// Schritt 5: Instantiiere die Template Dependency mit phi
					
					for( int attribute : attributesWithConstants ) {
						String constantValue = phi.getAttributeValue(attribute);
						String distinguishedSymbol = td.getConclusion().getElement(attribute).getValue();
						
						// instantiiere Konklusion
						td.getConclusion().setElement(attribute, new ConstantElement(constantValue));
						// propagiere Instantiierung in die Praemissen
						for( int i=0; i<td.getPremiseCount(); ++i ) {
							TemplateDependencyRow premise = td.getPremise(i);
							if( premise.getElement(attribute).isDistinguishedSymbol() && premise.getElement(attribute).getValue().equals(distinguishedSymbol) ) {
								premise.setElement(attribute, new ConstantElement(constantValue));
							}
						}
					}
					TDPhis.add(td);
				}
			}
		}		
		
		return TDPhis;
	}
	
	/**
	 * Diese Methode generiert die Menge Pot_Sig aus den mit den potentiellen Geheimnissen instantiierten Template Dependencies td_phi.
	 * Die Menge Pot_Sig entsteht dadurch, dass die mit den potentiellen Geheimnissen instantiierten Template Dependencies weiterhin mit
	 * der aktuellen Datenbankinstanz instantiiert werden.
	 * Anschliessend werden diejenigen Template Dependencies wieder entfernt, die direkt ein potentielles Geheimnis ueberdecken.
	 * Weiterhin werden evtl. doppelt vorkommenden Praemissen aus den Template Dependencies entfernt.
	 * Wichtig: Die Konklusionen der Template Dependencies werden NICHT instantiiert, da diese im weiteren Verlauf ohnehin nicht mehr 
	 * 			benoetigt werden.
	 * 
	 * @param tdPhi TD[phi]: Menge der Template Dependencies aus SigmaPlusMin, die mit den potentiellen Geheimnissen instantiiert wurden
	 * @param appDB Applikationsdatenbank, aus der die aktuelle Instanz ausgelesen werden soll
	 * @return		Pot_Sig: Menge der Template Dependencies aus SigmaPlusMin, die sowohl mit den potentiellen Geheimnissen als auch mit 
	 * 						 der aktuellen Datenbankinstanz instantiiert wurden
	 * @throws DatabaseException
	 */
	public static List<TemplateDependency> generatePot_Sig(List<TemplateDependency> tdPhi, AppDatabase appDB) throws DatabaseException {
		return generatePot_Sig(tdPhi, appDB, null);
	}
	
	public static List<TemplateDependency> generatePot_Sig(List<TemplateDependency> tdPhi, AppDatabase appDB, List<Integer> attributes) throws DatabaseException {
		// instantiiere alle Template Dependencies
		List<TemplateDependency> instantiatedDependencies =  new LinkedList<TemplateDependency>();
		for(TemplateDependency td : tdPhi) {
			// instanziiere die Template Dependency mit der Datenbankinstanz
			instantiatedDependencies.addAll(instantiateTemplateDependencyWithDatabaseInstance(td, appDB, attributes));
		}
		
		// entferne evtl. doppelt vorkommende Praemissen aus den Template Dependencies
		for( TemplateDependency td : instantiatedDependencies ) {
			for( int i=0; i<td.getPremiseCount(); ++i ) {
				TemplateDependencyRow p_i = td.getPremise(i);
				// vergleiche Praemisse mit allen folgenden Praemissen
				int j = i+1;
				while( j<td.getPremiseCount() ) {
					TemplateDependencyRow p_j = td.getPremise(j);
					if( p_i.equals(p_j) ) {
						// Praemisse doppelt vorhanden, Duplikat loeschen
						td.removePremise(j);
						// j nicht erhoehen, da durch das Loeschen der Praemisse die folgenden Praemissen um eins nach vorne rutschen
					}
					else {
						++j;
					}
				}
			}
		}
		
		return instantiatedDependencies;
	}
	
	private static List<TemplateDependency> instantiateTemplateDependencyWithDatabaseInstance(TemplateDependency td, AppDatabase appDB, List<Integer> attributes) throws DatabaseException {
		SQLDatabase db = appDB.getSQLDatabase();
		String[] columnNames = db.getColumnNames(td.getRelationname());
		
		// Attributfilter anwenden (falls vorhanden)
		if( attributes != null ) {
			String[] filteredColumnNames = new String[attributes.size()];
			int i = 0;
			for( int attribute : attributes ) {
				if( attribute < 0 || attribute > columnNames.length )
					throw new DatabaseException("Error while instantiating template dependency: Illegal attribute filter.", null);
				filteredColumnNames[i++] = columnNames[attribute];
			}
			columnNames = filteredColumnNames;
		}
		
		if( columnNames.length != td.getAttributeCount() ) {
			throw new DatabaseException("Error while instantiating template dependency: Number of attributes of the predicate doesn't match the number of columns in the database.", null);
		}
		
		String inner_query = "";
		// SELECT
		inner_query += "SELECT ";
		// Praemissen
		String seperator = "";
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				if( td.getPremise(i).getElement(attribute).isNonDistinguishedSymbol() && td.nonDistinguishedSymbolOccursOnlyOnce((NonDistinguishedSymbolElement) td.getPremise(i).getElement(attribute)) ) {
					// nicht ausgezeichnetes Symbol, fuer das keine Gleichheitsbedingung existiert -> keine Instantiierung!
					inner_query += seperator + "'" + td.getPremise(i).getElement(attribute).getValue() + "'" + " as " + columnNames[attribute] + i;
				}
				else {
					inner_query += seperator + "t" + i +"." + columnNames[attribute] + " as " + columnNames[attribute] + i;
				}
				seperator = ", ";
			}
		}
		// Konklusion
		for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
			inner_query += ", ";
			// Konklusion enthaelt nur ausgezeichnete Symbole oder Konstanten
			inner_query += "tc." + columnNames[attribute] + " as " + columnNames[attribute] + "c";
		}
		// FROM
		inner_query += " FROM ";
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			if( i>0 )
				inner_query += ", ";
			inner_query += td.getRelationname() + " t" + i;
		}
		inner_query += ", " + td.getRelationname() + " tc";
		// WHERE
		String inner_where = "";
		
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				TemplateDependencyElement elem = td.getPremise(i).getElement(attribute);
				
				if( elem.isConstant() ) {
					if( ! inner_where.equals("") )
						inner_where += " AND ";
					inner_where += "t" + i + "." + columnNames[attribute] + "='" + elem.getValue() + "'";
				}
				else if( elem.isDistinguishedSymbol() && td.getDistinguishedSymbolPositions().get(elem.getValue()).getSecondElement().size() >= 2 ) {
					// ausgezeichnete Variable, die ausser an der aktuellen Position noch an mind. einer weiteren Position in den Praemissen vorkommt (und definitionsgemaeß noch in der Konklusion -> >2)
					// ermittle Positionen der anderen Vorkommen
					String condition = "";
					Tuple<Integer, Set<Integer>> positions = td.getDistinguishedSymbolPositions().get(elem.getValue());
					for( Integer rowIndex : positions.getSecondElement() ) {
						if( rowIndex > i && rowIndex != -1 ) {
							// alle anderen Vorkommen wurden bereits in vorherigen Iterationen bereits beruecksichtigt
							if( ! condition.equals("") )
								condition += " AND ";
							
							// Symbole sind getypt -> in beiden Teilen der Bedingung steht der selbe Attributname
							condition += "t" + i + "." + columnNames[attribute] + "=" + "t" + rowIndex + "." + columnNames[attribute];
						}
						if( rowIndex == -1 ) {
							// Konklusion
							if( ! condition.equals("") )
								condition += " AND ";
							
							condition += "t" + i + "." + columnNames[attribute] + "=" + "tc." + columnNames[attribute];
						}
					}
					if( ! condition.equals("") ) {
						if( ! inner_where.equals("") )
							inner_where += " AND ";
						inner_where += condition;
					}
				}
				else if( elem.isNonDistinguishedSymbol() && ! td.nonDistinguishedSymbolOccursOnlyOnce((NonDistinguishedSymbolElement) elem) ) {
					// nicht ausgezeichnete Variable, die ausser an der aktuellen Position noch an mind. einer weiteren Position in den Praemissen vorkommt
					String condition = "";
					Tuple<Integer, Set<Integer>> positions = td.getMultipleOccurringNonDistinguishedSymbolPositions().get(elem.getValue());
					for( Integer rowIndex : positions.getSecondElement() ) {
						if( rowIndex > i ) {
							// alle anderen Vorkommen wurden bereits in vorherigen Iterationen bereits beruecksichtigt
							if( ! condition.equals("") )
								condition += " AND ";
							
							condition += "t" + i + "." + columnNames[attribute] + "=" + "t" + rowIndex + "." + columnNames[attribute];
						}
					}
					if( ! condition.equals("") ) {
						if( ! inner_where.equals("") )
							inner_where += " AND ";
						inner_where += condition;
					}
				}
			}
		}
		// Konklusion
		for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
			TemplateDependencyElement elem = td.getConclusion().getElement(attribute);
							
			if( elem.isConstant() ) {
				if( ! inner_where.equals("") )
					inner_where += " AND ";
				inner_where += "tc." + columnNames[attribute] + "='" + elem.getValue() + "'";
			}
		}
		if( ! inner_where.equals("") ) {
			inner_query += " WHERE " + inner_where;
		}
		
		
		String outer_query = "SELECT DISTINCT ";
		// Praemissen
		seperator = "";
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				outer_query += seperator + columnNames[attribute] + i;
				seperator = ", ";
			}
		}
		// Konklusion
		for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
			outer_query += ", ";
			outer_query += columnNames[attribute] + "c";
		}
		outer_query += " FROM (" + inner_query + ") ";
		String where = "";
		for( int i=0; i<td.getPremiseCount(); ++i ) {
			String rowCondition = "";
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				if( td.getConclusion().getElement(attribute).isConstant() ) {
					if( rowCondition != "" )
						rowCondition += " AND ";
					rowCondition += columnNames[attribute]+i + "='" + td.getConclusion().getElement(attribute).getValue() + "'";
				}
			}
			if( ! rowCondition.equals("") ) {
				if( ! where.equals("") )
					where += " OR ";
				where += "(" + rowCondition + ")";
			}
		}
		if( ! where.equals("") ) {
			outer_query += " WHERE NOT(" + where + ")";
		}
		
		SQLTable result = db.query(outer_query);
		// wandle Ergebnis in Template Dependencies um
		List<TemplateDependency> instantiatedDependencies = new LinkedList<TemplateDependency>();
		Iterator<String[]> iterator = result.iterator();
		while( iterator.hasNext() ) {
			String[] row = iterator.next();
			TemplateDependency instantiatedDep = new TemplateDependency(td);
			for( int i=0; i<instantiatedDep.getPremiseCount(); ++i ) {
				int j = 0;
				for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
					TemplateDependencyElement elem = instantiatedDep.getPremise(i).getElement(attribute);
					if( elem.isConstant() ) {
						// keine Instanziierung!
					}
					else if( elem.isNonDistinguishedSymbol() && td.nonDistinguishedSymbolOccursOnlyOnce((NonDistinguishedSymbolElement) elem) ) {
						// keine Instantziierung!
						instantiatedDep.getPremise(i).getElement(attribute).setValue("*");
					}
					else {
						instantiatedDep.getPremise(i).setElement(attribute, new ConstantElement(row[i*instantiatedDep.getAttributeCount() + j]));
					}
					++j;
				}
			}
			// Konklusion
			int j = 0;
			for( int attribute=0; attribute<td.getAttributeCount(); ++attribute ) {
				TemplateDependencyElement elem = instantiatedDep.getConclusion().getElement(attribute);
				if( elem.isConstant() ) {
					// keine Instantziierung!
				}
				else {
					instantiatedDep.getConclusion().setElement(attribute, new ConstantElement(row[td.getPremiseCount()*instantiatedDep.getAttributeCount() + j]));
				}
				++j;
			}
						
			instantiatedDependencies.add(instantiatedDep);
		}
		
		return instantiatedDependencies;
	}
}
