package server.censor.precqe;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.concurrent.TimeUnit;
import java.util.SortedSet;

import notification.serverToClient.PreCQETimeAndNodesNotification;

import org.apache.log4j.Logger;



import exception.DatabaseException;
import exception.InstancePrunedException;
import exception.NoInstanceFoundException;
import exception.UnsupportedFormulaException;

import server.database.NewMaintenanceDatabase;
import server.database.OracleSQLAppDatabase;
import server.database.cache.AppDatabaseSchemaCache;
import server.database.cache.Attribute;
import server.parser.Formula;
import server.parser.FormulaToPLNF;
import server.parser.node.ConjunctorNode;
import server.parser.node.DisjunctorNode;
import server.parser.node.ExistentialQuantifiedNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.VariableNode;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;


public class PreCQE {
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.censor.precqe");
	
	// Database access.
	private AppDatabaseSchemaCache appDBSchema;
	private MarkedOracleSQLAppDatabase markedDB;
	private NewMaintenanceDatabase maintenanceDB;
	
	// Current instance information.
	private Stack<TreeNode<Integer>> todoNodes;
	private int nodeCount;
	private int bestLieCount;
	private int foundSolution;
	private Map<String,Map<String,List<String>>> dictionary;
	
	// User information.
	private List<ConfidentialityPolicyItem> confidentialityPolicy;
	private List<IdAndFormula> prior;
	private List<IdAndFormula> schemaConstraints;

	
	/**
	 * Erzeugt eine Instanz des PreCQE-Algorithmus. Fuer den uebergebenen Benutzer kann eine inferenzsichere Instanz
	 * durch Aufruf der Methode {@link #createInferenceProofInstance()} erzeugt werden.
	 * 
	 * @param db Applikationsdatenbank, die inferenzsicher gemacht werden soll.
	 * @param maintenanceDB Maintenancedatenbank, aus der die Dictionaries fuer "erfundene" Konstanten verwendet werden.
	 * @param identifier Eindeutiger Bezeichner fuer einen Benutzer (meist dessen ID).
	 * @throws DatabaseException Tritt bei Datenbankproblemen auf.
	 */
	public PreCQE( OracleSQLAppDatabase db, NewMaintenanceDatabase maintenanceDB, String identifier, List<ConfidentialityPolicyItem> confPol, List<IdAndFormula> prior, List<IdAndFormula> schemaConstraints ) throws DatabaseException {
		this.appDBSchema = new AppDatabaseSchemaCache(db.getSQLDatabase(), maintenanceDB);
		this.markedDB = new MarkedOracleSQLAppDatabase(db, identifier, this.appDBSchema);
		this.maintenanceDB = maintenanceDB;
		
		this.nodeCount = 0;
		this.bestLieCount = Integer.MAX_VALUE;
		this.foundSolution = 0;
		this.dictionary = null;	// Wird erst beim Erstellen der inferenzsicheren Instanz erzeugt.
		
		this.confidentialityPolicy = confPol;
		this.prior = prior;
		this.schemaConstraints = schemaConstraints;
	}
	
	/**
	 * Erzeugt die temporaeren Tabellen zur Speicherung der inferenzsicheren Loesung.
	 * @throws DatabaseException
	 */
	public void init() throws DatabaseException {
		this.markedDB.createTemporaryTables();
	}
	
	/**
	 * Loescht die tempoeren Tabellen, die durch {@link #init()} erstellt wurden.
	 * @throws DatabaseException
	 */
	public void cleanup() throws DatabaseException {
		this.markedDB.removeTemporaryTables();
	}
	
	/**
	 * Erzeugt eine inferenzsichere Instanz fuer den im Kontruktor angegebenen Benutzer.
	 * 
	 * @throws DatabaseException Tritt bei Datenbankverbindungsproblemen auf.
	 * @throws UnsupportedFormulaException Tritt auf, falls eine Formel (Constraint) nicht range-restricted ist.
	 * @throws NoInstanceFoundException Tritt auf, falls keine inferenzsichere Instanz erzeugt werden konnte.
	 * @throws IOException 
	 */
	public PreCQETimeAndNodesNotification createInferenceProofInstance() throws DatabaseException, UnsupportedFormulaException, NoInstanceFoundException, IOException {
		this.cleanup();
		// Zeit messen
		long startTime = System.currentTimeMillis();
		this.init();
		
		// Constraints = Datenbankconstraints und Negation der ConfidentialityPolicy und Negation des Vorwissens.
		ArrayList<Formula> constraints = new ArrayList<Formula>();
		for ( ConfidentialityPolicyItem item : this.confidentialityPolicy ) {
			Formula constraint = (Formula)item.getFormula().clone();
			constraint.negate();
			FormulaToPLNF.plnf( constraint );
			constraints.add( constraint );
		}
		for ( IdAndFormula item : this.prior ) {
			Formula constraint = (Formula)item.getFormula().clone();
			FormulaToPLNF.plnf( constraint );
			constraints.add( constraint );
		}
		for ( IdAndFormula item : this.schemaConstraints ) {
			Formula constraint = (Formula)item.getFormula().clone();
			FormulaToPLNF.plnf( constraint );
			constraints.add( constraint );
		}
		
		// Fuer jede aktive Relation die Dictionary-Eintraege bestimmen.		
		this.dictionary = new HashMap<String,Map<String,List<String>>>();
		for ( Formula constraint : constraints ) {
			for ( String relation : constraint.getRelationnames() ) {
				Map<String, List<String>> attributes = new HashMap<String, List<String>>();
				if ( !this.dictionary.containsKey(relation) ) {
				   for ( Attribute attribute : appDBSchema.getRelation(relation) ){
					     attributes.put( attribute.getName(), this.maintenanceDB.getDictionary().getValues( relation, attribute.getName() ));
				   }
				   this.dictionary.put( relation, attributes );
				}
			}
		}
			
		// Create root node and add to todo.
		this.todoNodes = new Stack<TreeNode<Integer>>();
		TreeNode<Integer> root = new TreeNode<Integer>(this.nodeCount, constraints, 0);
		this.todoNodes.push( root );
		
		// Now iterate over the list of nodes that need to be examined.
		// If no node is left the algorithm has finished.
		while ( !this.todoNodes.isEmpty() ) {
			this.ground( this.todoNodes.pop() );
		}
		
		// Cleanup database.
		this.markedDB.removeTemporaryColumns();
		
		// Zeitmessung abschliessen
		long endTime = System.currentTimeMillis();
		long time = endTime-startTime;
		
		String processingTime = String.format("%d min, %d sec", 
			    TimeUnit.MILLISECONDS.toMinutes(time),
			    TimeUnit.MILLISECONDS.toSeconds(time) - 
			    TimeUnit.MINUTES.toSeconds(TimeUnit.MILLISECONDS.toMinutes(time))
			);

		// Create answer for the client.
		PreCQETimeAndNodesNotification notification = new PreCQETimeAndNodesNotification(bestLieCount, this.nodeCount, processingTime);
		return notification;
	}
	
	/**
	 * Rekursiver Branch & Bound Algorithmus zur Erzeugung einer inferenzsicheren Datenbankinstanz.
	 * 
	 * Erstellt eine Liste mit verletzten Constraints. Allquantifizierte verletzte Constraints werden umgewandelt, indem
	 * diejenigen grundierten Formeln gesucht werden, die die allquantifierte Formel erfuellen wuerden. Diese Formeln
	 * werden anstelle der allquantifizierten Constraints in die Menge der verletzten Constraints aufgenommen. Nach
	 * weiterer Vereinfachung dieser Menge durch {@link #simplification(List)} werden folgende Faelle entschieden:
	 * 
	 * 1. Wenn in einem grundiertem verletzten Constraint alle Atome markiert sind, dann kann keine gueltige Loesung
	 *    erzeugt werden, durch die dieses Constraint erfuellt werden koennte. Die Rekursion wird mittels {@link #prune(TreeNode)}
	 *    beendet.
	 * 2. Wenn in einem grundiertem verletzten Constraint genau ein Atom noch nicht markiert ist, dann wird dieses Atom
	 *    mittels {@link #mark(TreeNode, Node)} markiert und rekursiv {@link #ground(TreeNode)} aufgerufen.
	 * 3. Wenn in einem verletzten (nicht zwangslaeufig grundiertem) Constraint mindestens ein unmarkiertes grundiertes Atom
	 *    existiert, dann wird durch {@link #split(TreeNode, Node)} eine binaere Verzweigung im Baum erzeugt (Branch), bei dem
	 *    im einem Teilbaum das Atom und im anderen Teilbaum die Negation des Atoms markiert wird.
	 * 4. In allen anderen Faellen wird {@link #caseDifferentiation(TreeNode, List, List)} aufgerufen, das eine Variable
	 *    eines Existenzquantors mit der aktiven Domaene und einem Wert aus dem Dictionary grundiert und dadurch wie 3.
	 *    mehrere neue Teilbaeume erzeugt, die alle rekursiv weiterverarbeitet werden.
	 * 
	 * Bei jeder Markierung durch {@link #mark(TreeNode, Node)} kann es zu Luegen kommen, weshalb eine globale Schranke
	 * mit der bisher besten Luegenanzahl vorgehalten wird, um ggf. die Rekursion in Teilbaeumen abzubrechen, die keine
	 * bessere Loesung mehr erzeugen koennen (Bound).
	 * 
	 * @param v Aktuell bearbeiteter Knoten. Enthalt die zu bearbeitenden Constraints.
	 * @throws DatabaseException Tritt bei Datenbankverbindungsproblemen auf.
	 * @throws UnsupportedFormulaException Tritt auf, falls eine Formel (Constraint) nicht range-restricted ist.
	 * @throws NoInstanceFoundException Tritt auf, falls keine inferenzsichere Instanz erzeugt werden konnte.
	 */
	private void ground( TreeNode<Integer> v ) throws DatabaseException, UnsupportedFormulaException, NoInstanceFoundException {
		logger.debug("Examine node "+v);
		// Set marker (if any).
		Node mark = v.getMark();
		if ( mark != null ) {
			try {
				this.mark( v, mark );
			} catch (InstancePrunedException e) {
				// Too many lies in this subtree => abort
				logger.info("Node "+v.getKey()+" has too many lies");
				return;
			}
		}
		
		// Liste mit verletzten Constraints aufbauen.
		List<Formula> violatedConstraints = markedDB.getViolatedConstraints( v.getConstraints() );
		logger.debug("Violoated constraints: "+violatedConstraints);
		
		// Vorlaeufige Optimierung: Bestimme nur eine bestimmte Anzahl von inferenzsicheren Loesungen
		// und verwende daraus die beste.
//		if( this.foundSolution >= 1 ){
//			return;
//		}
		
		// Falls keine Constraints verletzt wird, bricht der Algorithmus an dieser Stelle ab.
		if ( violatedConstraints.isEmpty() ) {
			saveValidSolution(v);
			return;
		}
		
		// Iteriere ueber alle verletzten Constraints und suche diejenigen, die mit einem
		// Allquantor beginnen. Baue neue Liste mit verletzten Constraints auf, die maximal
		// noch Existenzquantoren enthalten (Zeile 2.3.1 bis 2.3.7).
		List<Formula> violatedExistentialConstraints = new ArrayList<Formula>();
		for ( Formula constraint : violatedConstraints ) {
			// Tiefe Kopie des Constraint anlegen, damit die urspruenglichen Constraints nicht veraendert werden.
			constraint = (Formula)constraint.clone();
			
			if ( constraint.getRootChild().isUniversalQuantifiedNode() ) {
				// Idee: FORALL X, Y (NOT R(X,Y) OR NOT P(X,Y)) (Denial Constraint) soll true sein
				//       => die Negation davon darf nicht wahr werden.
				// Negation: NOT FORALL X, Y (NOT R(X,Y) OR NOT P(X,Y)) = EXISTS X,Y (R(X,Y) AND P(X,Y))
				// Negation darf nicht wahr werden => finde alle a,b, fuer die R(a,b) AND P(a,b) gilt.
				// Fuege die Negation der gefundenen Instanzen von R(X,Y) AND P(X,Y) als Ersatz fuer die
				// allquantifizierte Formel als Constraints ein.
				
				// Negiere die allquantifizierte Formel und bilde die PLNF (Zeile 2.3.1, 2.3.2.1)
				constraint.negate();
				FormulaToPLNF.plnf(constraint);
				
				// Die resultierende Formel faengt mit einem (oder mehreren) Existenzquantor an.
				// Diese werden nun entfernt (Zeile 2.3.2.2) und damit werden die Variablen offen.
				ExistentialQuantifiedNode currentExistentialQuantor = (ExistentialQuantifiedNode)constraint.getRootChild();
				while ( currentExistentialQuantor.getQuantifiedFormula().isExistentialQuantifiedNode() ) {
					currentExistentialQuantor = (ExistentialQuantifiedNode)currentExistentialQuantor.getQuantifiedFormula();
				}
				constraint.setRootChild( currentExistentialQuantor.getQuantifiedFormula() );
				
				// Offene Anfrage stellen, welche Konstanten das umgewandelte Constraint erfuellen.
				// Diese Formeln sind der Grund dafuer, dass das urspruengliche Constraint verletzt wurde.
				// Zeile: 2.3.3
				List<Formula> violatingFormulas = markedDB.evalPosMarkedComplete( constraint );
				
				// Variablen so bennen, dass sie eine Formel eindeutig identifizieren. Die Identifikation wird
				// in Zeile 5.1 verwendet, um eine Formal anhand der Variable wiederzufinden.
				// Zeile 2.3.4
				this.standardizeVariables( violatingFormulas );

				// Gefundene Formeln negieren und PLNF bilden (Negation vor die Atome ziehen).
				// Optimierung: Wenn Variante (andere Variablennamen) der Formel bereits in der Menge der Constraints ist,
				//              dann nicht neu hinzufuegen.
				//				Wenn eine Instanz einer Formel bereits vorhanden ist, dann wird die allgemeinere Form
				//              (nicht instanziiert) ebenfalls erfuellt (R(a,b) => EXISTS X,Y R(X,Y)).
				// Zeile 2.3.5
				for ( Formula formula : violatingFormulas ) {
					formula.negate();
					FormulaToPLNF.plnf(formula);
					// FIXME: Optimierung pruefen
				}
				
				// Die Literale jetzt zur neuen Liste der verletzten Constraints hinzufuegen (Zeile 2.3.6).
				violatedExistentialConstraints.addAll( violatingFormulas );
			} else {
				// Die Formel hat keinen Allquantor an der ersten Stelle => direkt in die
				// neue Liste der verletzten Constraints aufnehmen (Zeile 2.3.6).
				violatedExistentialConstraints.add( constraint );
			}
		}
		
		// An dieser Stelle enthaelt violatedExistentialConstraints alle verletzten Constraints.
		// Insbesondere enthaelt kein Constraint mehr einen Allquantor (Denial Constraints wurden aufgeloest,
		// Existential Constraints enthalten sowieso keine, der Allquantor bei den Tuple-Generating Dependencies wurde aufgeloest)
		
		// Unnoetige Constraints entfernen/verkleinern lassen (2.3.8).
		List<Formula> simplifiedViolatedExistentialConstraints = this.simplification(violatedExistentialConstraints);
		
		// Liste der verletzten Constraints durchgehen und pruefen, welcher
		// Fall vorliegt.
		for ( Formula constraint : simplifiedViolatedExistentialConstraints ) {
			// Markierte Atome innerhalb des Constraints herausfinden.
			List<Node> unmarked = markedDB.unmarked( constraint );
			
			if ( constraint.getVariables().isEmpty() ) {
				// Grundiertes Constraint.
				if ( unmarked.size() == 0 ) {
					// Zeile 2.3.9
					// Alle Literale des grundierten Constraint sind markiert, aber es wird noch verletzt
					// (ansonst waere es in 2.1 nicht in die Menge aufgenommen worden). Es gibt keine Moeglichkeit
					// mehr das Constraint zu erfuellen => Ast im Baum abschneiden.
					try {
						this.prune(v);
					} catch (InstancePrunedException e) {
						// Expliziter prune()-Aufruf , Exception ignorieren.
					}
					return;
				} else if ( unmarked.size() == 1 ) {
					// Zeile 2.3.10
					Node literal = this.chooseLiteral(unmarked);
					v.setMark( literal );
					this.todoNodes.push( v );
					return;
				}
			}
			
			// Alle Constraints.
			if ( unmarked.size() > 0 ) {
				// Zeile 2.3.11
				this.split( v, this.chooseLiteral(unmarked) );
				return;
			}
		}
		
		// Keiner der vorherigen Faelle ist zugetroffen (alle (auch nicht grundierten) Literale sind markiert) 
		//=> CASE
		// Zeile 2.3.12		
		this.caseDifferentiation(v, violatedExistentialConstraints, simplifiedViolatedExistentialConstraints);
	}
	
	/**
	 * Vereinfacht die noch verletzten Constraints, indem bereits durch Markierung erfuellte Teile eines Constraints
	 * geloescht werden. Dadurch bleibt lediglich der noch nicht erfuellte Teil eines Constraints uebrig.
	 * 
	 * @param currentConstraints Liste von Constraints, die vereinfacht werden soll.
	 * @return Liste der vereinfachten Constraints.
	 * @throws UnsupportedFormulaException Wird bei fehlerhaften Formula-Objekten ausgeloest.
	 * @throws DatabaseException Tritt bei Datenbankverbindungsproblemen auf.
	 */
	private List<Formula> simplification( List<Formula> currentConstraints ) throws UnsupportedFormulaException, DatabaseException {
		// Erstelle tiefe Kopie aller Constraints (FIXME: teuer, kann mann man vielleicht irgendwie drauf verzichten?)
		List<Formula> simpConstraints = new ArrayList<Formula>();
		for ( Formula constraint : currentConstraints ) {
			simpConstraints.add( (Formula)constraint.clone() );
		}
		
		// Alle Konjunktionen und Disjunktionen in der Formel suchen und auf Resolution bzw. Subsumption pruefen.
		for ( Formula constraint : simpConstraints ) {
			this.simplifyFormula( constraint );
		}
		
		return simpConstraints;
	}
	
	/**
	 * Diese Methode besucht rekursiv alle Knoten im Syntaxbaum. Bei Konjunktion und Disjunktion wird
	 * geprueft, ob Resolution bzw. Subsumption angewendet mit Hilfe der Marker werden kann.
	 * Teilbaeume, die durch Resolution/Subsumption wegfallen, werden nicht besucht.
	 * @param formula
	 * @throws DatabaseException 
	 * @throws UnsupportedFormulaException 
	 */
	private void simplifyFormula( Node formula ) throws UnsupportedFormulaException, DatabaseException {
		Node newChild = null;
		
		if ( formula.isConjunctorNode() ) {
			newChild = new ConjunctorNode();
			
			for ( Node junctor : ((ConjunctorNode)formula).getOperands() ) {
				if ( junctor.isAtomPredicateNode() ) {
					char marker = markedDB.marker( junctor );
					if ( marker == 'k' || marker == 'a' ) {
						// 3.1.1.1 Resolution: Atom nicht zur neuen Konjunktion hinzufuegen.
						continue;
					} else if ( marker == 'r' || marker == 'l' ) {
						// 3.1.1.2 Subsumption: Nur Atom relevant.
						newChild = junctor;
						break;
					} else {
						// Keine Markierung: Als Kind übernehmen.
						newChild.addChild( junctor );
					}
				} else if ( junctor.isNegationNode() ) {
					// Nach einger Negation muss laut PLNF ein Atom stehen => Literal.
					char marker = markedDB.marker( junctor );
					if ( marker == 'r' || marker == 'l' ) {
						// 3.1.1.1 Resolution: Atom nicht zur neuen Konjunktion hinzufuegen.
						continue;
					} else if ( marker == 'k' || marker == 'a' ) {
						// 3.1.1.3 Subsumption: Nur Negation des Atoms relevant.
						newChild = junctor;
						break;
					} else {
						// Keine Markierung: Als Kind übernehmen.
						newChild.addChild( junctor );
					}
				} else {
					// Kein Atom und keine Negation: Als Kind uebernehmen.
					newChild.addChild( junctor );
				}
			}
		} else if ( formula.isDisjunctorNode() ) {
			newChild = new DisjunctorNode();
			
			for ( Node junctor : ((DisjunctorNode)formula).getOperands() ) {
				if ( junctor.isAtomPredicateNode() ) {
					char marker = markedDB.marker( junctor );
					if ( marker == 'r' || marker == 'l' ) {
						// 3.1.1.1 Resolution: Atom nicht zur neuen Disjunktion hinzufuegen.
						continue;
					} else if ( marker == 'k' || marker == 'a' ) {
						// 3.1.1.2 Subsumption: Nur Atom relevant.
						newChild = junctor;
						break;
					} else {
						// Keine Markierung: Als Kind übernehmen.
						newChild.addChild( junctor );
					}
				} else if ( junctor.isNegationNode() ) {
					// Nach einger Negation muss laut PLNF ein Atom stehen => Literal.
					char marker = markedDB.marker( junctor );
					if ( marker == 'k' || marker == 'a' ) {
						// 3.1.1.1 Resolution: Atom nicht zur neuen Disjunktion hinzufuegen.
						continue;
					} else if ( marker == 'r' || marker == 'l' ) {
						// 3.1.1.3 Subsumption: Nur Negation des Atoms relevant.
						newChild = junctor;
						break;
					} else {
						// Keine Markierung: Als Kind übernehmen.
						newChild.addChild( junctor );
					}
				} else {
					// Kein Atom und keine Negation: Als Kind uebernehmen.
					newChild.addChild( junctor );
				}
			}
		}
		
		if ( newChild != null ) {
			// Pruefen, ob neue Konjunktion/Disjunktion nur noch ein Kind enthaelt. Wenn ja, dann nur dieses
			// Kind ohne Konjunktion/Disjunktion uebernehmen.
			if ( newChild.isConnectorNode() && newChild.getChildrenNumber() == 1 ) {
				newChild = newChild.getChild(0);
			}
			
			// Kind durch neue Konjunktion bzw. Atom ersetzen.
			formula.replace( newChild );
			formula = newChild;
		}
		
		// Egal welchen Knotentyp wir haben, die Kinder muessen evtl. noch umgebaut werden.
		// Es wird eine Kopie der Kindliste verwendet, damit replace() (s.o.) die aktuelle
		// Kindliste modifizieren kann.
		for ( Node child: new ArrayList<Node>(formula.getChildren()) ) {
			this.simplifyFormula( child );
		}
	}
	
	private void split( TreeNode<Integer> v, Node literal ) throws DatabaseException, UnsupportedFormulaException, NoInstanceFoundException {
		// Zwei neue Kinder erzeugen, fuer die lieCount und Constraints identisch
		// mit ihrem Elter sind.
		TreeNode<Integer> leftChild = new TreeNode<Integer>( this.nodeCount + 1, new ArrayList<Formula>(v.getConstraints()), v.getLieCount() );
		TreeNode<Integer> rightChild = new TreeNode<Integer>( this.nodeCount + 2, new ArrayList<Formula>(v.getConstraints()), v.getLieCount() );
		this.nodeCount += 2;
		
		v.addChild(leftChild);
		v.addChild(rightChild);
		
		// Im rechten Kind negieren.
		Formula literalCopy = new Formula(literal.clone());
		literalCopy.negate();
		rightChild.setMark(literalCopy);
		this.todoNodes.push(rightChild);
		
		// Im linken Kind das Literal nicht negieren.
		leftChild.setMark(literal);
		this.todoNodes.push(leftChild);
	}

	
	
	private void caseDifferentiation( TreeNode<Integer> v, List<Formula> constraints, List<Formula> simplifiedConstraints ) throws NoInstanceFoundException, DatabaseException, UnsupportedFormulaException {
		// Eine existenzquantifizierte Formel auswaehlen (Zeile 5.1).
		Formula existentialConstraint = null;
		String existentialVariable = null;
		
		// Alle Variablennamen aus den simplifizierten Constraints extrahieren.
		List<String> vars = getAllVariableNames( simplifiedConstraints );
		
		for ( Formula constraint : constraints ) {
			if ( constraint.getRootChild().isExistentialQuantifiedNode() ) {
				ExistentialQuantifiedNode exConstraint = (ExistentialQuantifiedNode)constraint.getRootChild();
				
				// Pruefen, ob auch in der nicht simplizifizerten Menge enthalten,
				// da eine die Variable nur dann ersetzt werden soll, wenn sie nicht durch Simplifizierung verschwindet.
				// Die Variablennamen sind eindeutig und identifizieren Formeln.
				for ( String var : exConstraint.getQuantifiedVariables() ) {
					if ( vars.contains(var) ) {
						existentialVariable = var;
						break;
					}
				}
				
				if ( existentialVariable != null ) {
					existentialConstraint = constraint;
					break;
				}
			}
		}
		
		if ( existentialConstraint == null ) {
			throw new NoInstanceFoundException("case() found no EXISTS, this should be impossible. Check your code.");
		}
		
		
		// Aktive Domaene fuer Variable ausrechnen.
		Set<String> activeDomain = new HashSet<String>();
		Map<String, SortedSet<Integer>> activeDomainMapping = existentialConstraint.getRelationnamesAndAttributesContainingVar( existentialVariable );
		for ( Map.Entry<String, SortedSet<Integer>> entry : activeDomainMapping.entrySet() ) {
			List<Attribute> attributes = this.appDBSchema.getRelation( entry.getKey() );
			
			for ( Integer attributeIndex : entry.getValue() ) {
				activeDomain.addAll( attributes.get(attributeIndex).getActiveDomain() );
			}
		}
		
		// Domaenenwert fuer die Variable einsetzen.
		for ( String value : activeDomain ) {
			Formula constraint = this.createFormulaWithoutVariable(v, existentialConstraint, existentialVariable);
			
			HashMap<String,String> mapping = new HashMap<String,String>();
			mapping.put(existentialVariable, value);
			constraint.substituteVariables(mapping);
		}
		
		// Invent enthaelt ausschliesslich die "erfundenen" Konstanten,
		// insbesondere keine Werte aus der aktiven Domaene.
		Formula constraint = this.createFormulaWithoutVariable(v, existentialConstraint, existentialVariable);
		constraint.substituteVariable(this.appDBSchema, existentialVariable);
		
		// Put all children on todo list (reverse order, because it's a stack).
		ListIterator<TreeNode<Integer>> iter = v.getChildren().listIterator(v.getChildren().size());
		while ( iter.hasPrevious() ) {
			this.todoNodes.push( iter.previous() );
		}
	}
	
	/**
	 * Erzeugt ein neues Kind, bei dem der Existenzquantor des ausgewaehlten Constraint
	 * eine Variable weniger enthaelt. Die restlichen Constraints werden vom Vater v uebernommen.
	 * Das modifizierte Constraints wird zurueckgeliefert und muss noch instanziiert werden.
	 * @param v
	 * @param existentialConstraint
	 * @param var
	 * @param a
	 * @return
	 */
	private Formula createFormulaWithoutVariable( TreeNode<Integer> v, Formula existentialConstraint, String var ) {
		Formula newConstraint = null;
		
		// Tiefe Kopie des Quantors anlegen.
		ExistentialQuantifiedNode constraint = (ExistentialQuantifiedNode)existentialConstraint.getRootChild().clone();

		// Eine Variable aus dem Quantor entfernen.
		int index = 0;
		for ( VariableNode varNode : constraint.getVariableNodes() ) {
			if ( varNode.getVariable().equals(var) ) {
				break;
			}
			index++;
		}
		if ( index == constraint.getVariableNodes().size() ) {
			throw new RuntimeException("Variable that must exists was not found.");
		}
		constraint.getVariableNodes().remove(index);
		
		// Pruefen, ob Quantor nach Variablenloeschung ueberfluessig ist.
		if ( constraint.getQuantifiedVariables().isEmpty() ) {
			newConstraint = new Formula(constraint.getQuantifiedFormula());
		} else {
			newConstraint = new Formula(constraint);
		}

		// Constraints fuer neues Kind erstellen.
		List<Formula> aConstraints = new ArrayList<Formula>( v.getConstraints() );
		aConstraints.remove( existentialConstraint );
		aConstraints.add( newConstraint );

		// Kind erzeugen.
		TreeNode<Integer> child = new TreeNode<Integer>( this.nodeCount + 1, aConstraints, v.getLieCount() );
		v.addChild(child);
		this.nodeCount++;
		
		return newConstraint;
	}
	
	
	
	
	private void mark( TreeNode<Integer> v, Node literal ) throws DatabaseException, UnsupportedFormulaException, NoInstanceFoundException, InstancePrunedException {
		// Atom aus Literal extrahieren (Zeile 6.1)
		literal = literal.isRootNode() ? ((Formula)literal).getRootChild() : literal;
		Node atom = literal.isNegationNode() ? ((NegationNode)literal).getNegatedFormula() : literal;
		
		// Literal als Anfrage an die Datenbank stellen. Wird von jedem Fall von mark() benoetigt.
		boolean dbAnswer = markedDB.evalMarkedComplete( new Formula(atom.clone()) );
		
		if ( atom.equals(literal) ) {
			if ( dbAnswer == true ) {
				// Zeile 6.2
				markedDB.mark( v, literal, 'k' );
			} else {
				// Zeile 6.3
				markedDB.mark( v, literal, 'a' );
				v.setLieCount( v.getLieCount()+1 );
			}
		} else {
			if ( dbAnswer == true ) {
				// Zeile 6.4
				markedDB.mark( v, literal, 'r' );
				v.setLieCount( v.getLieCount()+1 );
			} else {
				// Zeile 6.5
				markedDB.mark( v, literal, 'l' );
			}
		}
		
		// Zeile 6.6
		if ( v.getLieCount() >= this.bestLieCount ) {
			logger.debug("Upper bound prune: "+v.getKey());
			prune( v );
		}
	}
	

	/**
	 * Schneidet einen Ast im Baum ab. Mittels Backtracking werden Knoten ohne
	 * Kinder ebenfalls entfernt. Alle Markierungen, die durch die betroffenen
	 * Teilbaeume gesetzt wurde, werden entfernt.
	 * 
	 * @param v
	 * @throws NoInstanceFoundException 
	 * @throws DatabaseException 
	 */
	private void prune( TreeNode<Integer> v ) throws NoInstanceFoundException, DatabaseException, InstancePrunedException {
		// In dieser Liste wird sich der Pfad gemerkt, der mittels Backtracking abgeschnitten wird.
		ArrayList<Integer> path = new ArrayList<Integer>();
		
		// Ausgehend vom aktuelle Knoten wandere solange im Baum Richtung Wurzel, wie genau
		// ein Kind existiert (Ast, der abgeschnitten werden kann)
		do {
			path.add( v.getKey() );
			// Markierungen entfernen, die im Knoten gesetzt worden sind.
			markedDB.unmark( String.valueOf(v.getKey()) );
			// Zum naechsten Knoten gehen.
			v = v.getParent();
		} while ( v != null && !v.isRoot() && v.getChildCount() == 1 );
		
		// Wenn v == null ist, dann hat es auch fuer die Wurzel keine Alternativen mehr gegeben
		// => es gibt keine Loesung.
		if ( v == null && this.foundSolution == 0 ) {
			throw new NoInstanceFoundException("Unable to determine inference proof instance for the given set of constraints");
		}
		
		// Vom aktuellen Knoten (der noch mind. 2 Kinder hat) den vorher bestimmten Ast abschneiden.
		if ( v != null ){ // wenn v null ist, hat es nur noch ein Kind
			Integer key = path.get(path.size()-1);
			Iterator<TreeNode<Integer>> iter = v.getChildren().iterator();
			while ( iter.hasNext() ) {
				TreeNode<Integer> node = iter.next();
				if ( node.getKey() == (int)key ) {
					iter.remove();
					break;
				}
			}
		}
		
		throw new InstancePrunedException("Current instance was pruned");
	}
	
	private void saveValidSolution( TreeNode<Integer> v ) throws DatabaseException, NoInstanceFoundException {
		logger.debug("Found new solution with "+v.getLieCount()+" lies.");
		
		// Gueltiges Ergebnis speichern, falls besser als bereits gefundenes.
		if ( v.getLieCount() < this.bestLieCount ) {
			this.bestLieCount = v.getLieCount();
			this.markedDB.copyTemporaryTableToBestTable();
		}
		
		this.foundSolution++;
		
		this.checkLocalLowerBound(v);
		
		// Auch gueltige Loesungen werden abgeschnitten! Schnipp, schnapp.
		try {
			this.prune(v);
		} catch (InstancePrunedException e) {
			// Explicit prune, ignore exception.
		}
	}
	
	private void checkLocalLowerBound( TreeNode<Integer> v ) {
		int lies = v.getLieCount();
		TreeNode<Integer> parent = v.getParent();
		
		// Ascend the tree and check for nodes with only one lie less.
		while ( parent != null && parent.getLieCount() >= lies-1 ) {
			// Delete all remaining children of parent from the todo list.
			while ( this.todoNodes.peek().getParent() == parent ) {
				TreeNode<Integer> delete = this.todoNodes.pop();
				logger.debug("Lower bound deletion: "+delete);
			}
			
			// Go upwards.
			parent = parent.getParent();
		}
	}
	
	/**
	 * Gibt die Menge aller in der Menge von Formeln verwendeten Variablen zurueck.
	 * @param formulas
	 * @return
	 */
	private List<String> getAllVariableNames( List<Formula> formulas ) {
		ArrayList<String> result = new ArrayList<String>();
		
		for ( Formula formula : formulas ) {
			result.addAll( formula.getVariables() );
		}
		
		return result;
	}
	
	// FIXME: stubs, implement me!
	private void standardizeVariables( List<Formula> formulas ) { return; }
	

	private Node chooseLiteral( List<Node> literals ) { return literals.get(0); }
}
