package server.censor.precqe;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import exception.DatabaseException;
import exception.UnsupportedFormulaException;

import server.database.OracleSQLAppDatabase;
import server.database.cache.AppDatabaseSchemaCache;
import server.database.sql.Column;
import server.database.sql.SQLDelete;
import server.database.sql.SQLInsert;
import server.database.sql.SQLSelect;
import server.database.sql.SQLTable;
import server.database.sql.SQLUpdate;
import server.database.sql.SQLWhereExpression;
import server.util.Tuple;
import server.parser.Formula;
import server.parser.Formatter.ConstantWithoutQuotesFormatter;
import server.parser.Formatter.FormulaFormatter;
import server.parser.node.AtomPredicateNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.parser.node.TermNode;

public class MarkedOracleSQLAppDatabase extends OracleSQLAppDatabase {
	
	public static String userSeparator = "$#";
	private FormulaFormatter formatter;	// Used to prevent adding " to Predicate-Constants in SQL-INSERT.
	private String tableSuffix;
	
	public MarkedOracleSQLAppDatabase(OracleSQLAppDatabase appDB, String identifier, AppDatabaseSchemaCache appDBSchema) throws DatabaseException {
		super( appDB.getSQLDatabase() );
		this.formatter = new ConstantWithoutQuotesFormatter();
		this.tableSuffix = userSeparator+identifier;
		this.toSQL.setTableSuffix( this.tableSuffix );
	}

	/**
	 * Diese Methode kopiert jede Tabelle der Datenbank und bennent sie mit einem Suffix, das den zugehoerigen 
	 * User identifiziert (_userid).
	 * Zusaetzlich werden zwei Spalten in die *_tmp Tabellen eingefuegt:
	 * - NodeKey: Gibt den Pfad zum Knoten im PreCQE-Baum an, in dem die Markierung in Marker gesetzt wurde.
	 * - Marker: Feld, um die Markierung (siehe {@link #mark(TreeNode, Node, char)}) zu speichern.
	 * @throws DatabaseException 
	 */
	public void createTemporaryTables() throws DatabaseException {
		for ( String relation : this.db.getTables() ) {
			// User-Tabellen ueberspringen.
			if ( relation.contains(userSeparator) ) {
				continue;
			}
			
			//Name der neuen Relation
			String tmpRelation = relation + this.tableSuffix;
			
			// Fuer jede Relation muss eine neue Attributliste angelegt werden.
			List<Tuple<String,String>> attributes = new ArrayList<Tuple<String,String>>();
			
			// Kopieren aller Attribute einer Tabelle	
			for ( Column column : this.db.getColumns(relation) ){
				attributes.add( new Tuple<String,String>(column.getName(), column.getType()) );
			}
			
			// Hinzufuegen der zwei neuen Spalten
			attributes.add( new Tuple<String,String>("NodeKey", "VARCHAR2(500)") );
			attributes.add( new Tuple<String,String>("Marker", "VARCHAR2(500)") );
			
			// Erstellen der Arbeitstabelle
			this.db.createNewTable( tmpRelation, attributes );
			// Erstellen der Tabelle fuer die beste Loesung
			this.db.createNewTable( tmpRelation + "_B", attributes);
			
			// Arbeitstabelle mit Inhalt fuellen
			String sql = "INSERT INTO " + tmpRelation + " (SELECT "+ relation +".*,'X','X' FROM " + relation + ")";
			this.db.dataManipulationQuery(sql);
		}
		
		// Cache aktualisieren.
		this.db.reloadSchemaInformation();
	}
	
	/**
	 * Copies the contents all temporary work-tables to the corresponding best-solution-tables.
	 * @throws DatabaseException
	 */
	public void copyTemporaryTableToBestTable() throws DatabaseException {
		for ( String relation : this.db.getTables() ) {
			//Name der neuen Relation
			String tmpSuffix = this.tableSuffix;
			
			// Nicht-Arbeitstabellen ueberspringen.
			if ( !relation.endsWith(tmpSuffix) ) {
				continue;
			}
			
			// Arbeitstabelle in Tabelle mit bester Loesung kopieren.
			this.db.dataManipulationQuery("TRUNCATE TABLE "+relation+"_B");
			String sql = "INSERT INTO " + relation+"_B" + " (SELECT "+ relation +".* FROM " + relation + ")";
			this.db.dataManipulationQuery(sql);
		}
	}
	
	/**
	 * Loescht die mittels {@link #createTemporaryTables()} angelegten Tabellen mit dem Suffix, 
	 * das den zugehoerigen User identifiziert.
	 * Vorher werden die Zeilen, die r oder l als Marker gesetzt haben, aus der Datenbank geloescht. 
	 * @throws DatabaseException 
	 */
	public void removeTemporaryColumns() throws DatabaseException {

		for ( String relation : this.db.getTables() ) {
			
			if( relation.endsWith(this.tableSuffix) || relation.endsWith(this.tableSuffix+"_B") ) {
				SQLDelete sql = this.db.newDelete();
				sql.or_where("Marker", "r").or_where("Marker", "l");
				sql.delete(relation);

				this.db.dropColumn( relation, "NodeKey" );
				this.db.dropColumn( relation, "Marker" );
			}
		}
	}
	
	/**
	 * Loescht alle benutzerspezifischen Tabellen (Tabellen mit Suffix _USERNAME) des aktuell
	 * fuer die Datenbank festgelegten Benutzers.
	 * 
	 * @throws DatabaseException
	 */
	public void removeTemporaryTables() throws DatabaseException {
		for ( String relation : this.db.getTables() ){
			if( relation.endsWith(this.tableSuffix) || relation.endsWith(this.tableSuffix+"_B") ) {
				this.db.dropTable(relation);
			}
		}
		
		this.db.reloadSchemaInformation();
	}

	/**
	 * Setzt die Markierung fuer ein Literal (bzw. dessen Atom) in der Datenbank.
	 * Das Atom erhalt zusaetzlich den Schluesselwert des Knotens als Markierungsschluessel.
	 * Dadurch besteht eine 1 zu 1 Beziehung zwischen Knoten und von ihm markierten Atomen.
	 * Wird beim Verwerfen von Loesungen (Backtracking) benoetigt.
	 * 
	 * Die folgenden Marker sind erlaubt:
	 * 'k' (keep):   Das Atom soll in der inferenzsicheren Datenbank behalten werden.
	 * 'a' (add):    Ein noch nicht in der Datenbank befindliches Atom soll in die inferenzsichere Datenbank eingefuegt werden.
	 * 'r' (remove): Ein bisher in der Datenbank befindliches Atom soll nicht in die inferenzsichere Datenbank uebernommen werden.
	 * 'l' (leave):  Das Atom befindet sich nicht in der Datenbank und soll auch nicht in die inferenzsichere Datenbank.
	 * 
	 * @param v Aktueller Knotem im PreCQE-Baum.
	 * @param literal NegationNode mit AtomPredicateNode als Kind oder direkt eine AtomPredicateNode.
	 * @param marker Marker, der fuer das in literal enthaltene Atom gesetzt werden soll.
	 * @throws UnsupportedFormulaException Diese Exception wird ausgeloest, wenn der Parameter literal kein Literal ist.
	 * @throws DatabaseException 
	 */
	public void mark( TreeNode<Integer> v, Node literal, Character marker ) throws UnsupportedFormulaException, DatabaseException {
		int count = -1;
		AtomPredicateNode atom = Node.getAtomFromLiteral(literal);
		String relationname = atom.getRelationname().toUpperCase()+this.tableSuffix;
		
		// Marker fuer das Atom setzen.
		if ( marker == 'r' || marker == 'k' ) {
			// Atom in DB => UPDATE
			SQLUpdate sql = this.db.newUpdate();
			sql.set("NodeKey", v.getKey()).set("Marker", marker.toString());
			this.setWhereFromAtom(sql, atom);
			count = sql.update( relationname );
		} else if ( marker == 'a' || marker == 'l' ) {
			// Atom nicht in DB => INSERT
			SQLInsert sql = this.db.newInsert();
			sql.set("NodeKey", v.getKey()).set("Marker", marker.toString());
			Iterator<Column> attributeIter = this.db.getColumns(relationname).iterator();
			for ( TermNode value : atom.getParameterNodes() ) {
				String key = attributeIter.next().getName();
				if ( value.isVariableNode() ) {
					throw new UnsupportedFormulaException(literal.toString());
				}
				
				// Use Formatter without quotes, because set() will set them in the correct manner.
				sql.set(key, value.toString(formatter));
			}
			count = sql.insert( relationname );
		}
		
		if ( count != 1 ) {
			throw new DatabaseException("INSERT/UPDATE failed, wrong number of elements ("+count+") where updated", null);
		}
	}
	
	/**
	 * Loescht Markierungen von Atomen, die einen Markierungsschluessel aus der uebergebenen Liste erhalten haben.
	 * 
	 * @param markerKeys
	 * @throws DatabaseException 
	 */
	public void unmark( String markerKeys ) throws DatabaseException {
		SQLUpdate sql1 = this.db.newUpdate();
		SQLDelete sql2 = this.db.newDelete();

		// markerKeys als Bindvariable hinzufuegen.
		List<Object> bindVars = new LinkedList<Object>();
		bindVars.add( markerKeys );
		sql1.setWhereBindVariables( bindVars );
		sql2.setWhereBindVariables( bindVars );
		
		// 1. Schritt: Marker 'r' und 'k' loeschen, weil die betroffenen Tupel zur Originalinstanz gehoeren.
		sql1.where("NodeKey=? AND (Marker='r' OR Marker='k')");
		sql1.set("NodeKey", "X").set("Marker", "X");
		
		// 2. Schritt: Marker 'a' und 'l' sind nicht in der Originalinstanz enthalten, deshalb muessen sie raus!
		sql2.where("NodeKey=? AND (Marker='a' OR Marker='l')");
		
		// Alle Tabellen mit _tmp Suffix ueberpruefen.
		for ( String table : this.db.getTables() ) {
			if ( table.endsWith(this.tableSuffix) ) {
				sql1.update(table);
				sql2.delete(table);
			}
		}
	}
	
	/**
	 * Gibt die Markierung eines Literals in der Datenbank zurueck.
	 * Wenn das Literal nicht markiert ist, wird ein Leerzeichen ' ' zurueckgegeben.
	 * 
	 * @param literal
	 * @return 'k', 'a', 'r', 'l' oder 'X' falls keine Markierung gesetzt ist.
	 * @throws UnsupportedFormulaException 
	 * @throws DatabaseException 
	 */
	public char marker( Node literal ) throws UnsupportedFormulaException, DatabaseException {
		AtomPredicateNode atom = Node.getAtomFromLiteral(literal);
		
		SQLSelect sql = this.db.newSelect();
		sql.select("Marker").from(atom.getRelationname()+this.tableSuffix);
		this.setWhereFromAtom(sql, atom);
		
		// FIXME: error handling?
		SQLTable result = sql.get();
		try {
			return (result.getFirstRow())[0].charAt(0);
		} catch ( IndexOutOfBoundsException exception ) {
			return 'X';
		} catch ( DatabaseException exception ) {
			return 'X';
		}
	}
	
	/**
	 * Gibt die Menge aller nicht markierten Literale innerhalb einer Formel zurueck.
	 * @param constraint
	 * @return
	 * @throws DatabaseException 
	 * @throws UnsupportedFormulaException 
	 */
	public List<Node> unmarked( Formula constraint ) throws UnsupportedFormulaException, DatabaseException {
		ArrayList<Node> unmarkedNodes = new ArrayList<Node>();
		this.unmarkedRecursion(unmarkedNodes, constraint);
		return unmarkedNodes;
	}

	/**
	 * Durchlaeuft rekursiv den Syntaxbaum einer Formel und fuegt der Liste
	 * unmarkierte Atome hinzu.
	 * 
	 * @param unmarkedNodes
	 * @param currentNode
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 */
	private void unmarkedRecursion( List<Node> unmarkedNodes, Node currentNode ) throws UnsupportedFormulaException, DatabaseException {
		if ( currentNode.isAtomPredicateNode() ) {
			// Nur grundierte Literale werden zurueckgegeben.
			if ( currentNode.getFreeVariables().size() > 0 ) {
				return;
			}
			char marker = this.marker( currentNode );
			if ( marker == 'X' ) {
				unmarkedNodes.add( currentNode );
			}
		}
		else if ( currentNode.isNegationNode() &&  ((NegationNode)currentNode).getNegatedFormula().isAtomPredicateNode()) {
			NegationNode negation = (NegationNode)currentNode;

			char marker = this.marker( negation.getNegatedFormula() );
			if ( marker == 'X' ) {
				unmarkedNodes.add( negation );
			}
		} else {
			for ( Node child : currentNode.getChildren() ) {
				this.unmarkedRecursion( unmarkedNodes, child );
			}
		}
	}
	
	/**
	 * Gibt eine Liste mit allen Constraints aus dem Parameter zurueck, die nicht in der Datenbank wahr sind.
	 * @param constraints
	 * @return
	 * @throws UnsupportedFormulaException 
	 * @throws DatabaseException 
	 */
	public List<Formula> getViolatedConstraints( List<Formula> constraints ) throws DatabaseException, UnsupportedFormulaException {
		ArrayList<Formula> violatedConstraints = new ArrayList<Formula>();
		
		for ( Formula constraint : constraints ) {
			if ( this.evalMarkedComplete(constraint) == false ) {
				violatedConstraints.add( constraint );
			}
		}
		return violatedConstraints;
	}
	
	
	/**
	 * Gibt wahr zurueck, wenn formula in der Datenbank wahr ist UND marker(formula) fuer alle vorkommenden Atome
	 * ungleich 'r' (remove) und 'l' (leave) sind.
	 * @param formula
	 * @return
	 * @throws UnsupportedFormulaException 
	 * @throws DatabaseException 
	 */
	public boolean evalMarkedComplete( Formula formula ) throws DatabaseException, UnsupportedFormulaException {
		this.toSQL.setMarkerMode(true);
		boolean result = this.evalComplete(formula);
		this.toSQL.setMarkerMode(false);
		return result;
	}
	
	/**
	 * Wertet offene Anfragen in der Datenbank aus und gibt eine Menge von Instanzen
	 * der uebergebenen Formel zurueck.
	 * 
	 * @param formula
	 * @return
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 */
	public List<Formula> evalPosMarkedComplete( Formula formula ) throws UnsupportedFormulaException, DatabaseException {
		this.toSQL.setMarkerMode(true);
		List<Formula> result = this.evalPosComplete(formula);
		this.toSQL.setMarkerMode(false);
		return result;
	}

	
	/**
	 * Setzt in der uebergebenen SQL-Anfrage den WHERE-Teil so, dass
	 * ein bestimmtes Atom ausgewaehlt wird.
	 * @param sql
	 * @param atom
	 * @throws DatabaseException
	 */
	private void setWhereFromAtom( SQLWhereExpression<?> sql, AtomPredicateNode atom ) throws DatabaseException {
		List<Column> columns = this.db.getColumns( atom.getRelationname() );
		Iterator<Column> columnIter = columns.iterator();
		
		for ( Node children : atom.getParameterNodes() ) {
			// Formatter ohne Anfuehrungszeichen, weil SQLFormatter bei where() schon Anfuehrungszeichen setzt.
			sql.where( columnIter.next().getName(), children.toString(formatter) );
		}
	}
}
