package server.censor.lying;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.List;

import server.censor.OpenCensor;
import server.core.Client;
import server.core.Server;
import server.parser.CompletenessSentenceFormula;
import server.parser.Formula;
import server.theoremprover.ProverNine;
import server.theoremprover.ProverResult;
import server.theoremprover.TheoremProver;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;
import exception.DatabaseException;
import exception.DictionaryTooSmallException;
import exception.ProverException;
import exception.UnsupportedFormulaException;



/**
 * Die Klasse OpenlyingCensor ist ein Zensor fuer die Auswertung offener Anfragen auf Basis der Luegen-Methode
 * bei bekannten potentiellen Geheimnissen und einer als vollstaendig angenommenen Datenbankinstanz.
 * Details zur Funktionsweise dieses Zensors koennen dem Paper "Controlled Query Evaluation with Open Queries for a
 * Decidable Relational Submodel" von Joachim Biskup und Piero Bonatti entnommen werden.
 * 
 * @author bring
 *
 */
public class OpenLyingCensor extends OpenCensor {
	
	private LyingCensor closedLyingCensor;
	
	/**
	 * Konstruktur: Erzeugt ein neues Objekt dieses Zensor-Typs
	 * @param user Der User, fuer den ein konkretes Objekt dieses Zensors Entscheidungen treffen soll.
	 */
	public OpenLyingCensor(Server server) {
		super("OpenLying", server);
	}

	/** 
	 *  Die Methode implementiert die (dynamische) kontrollierte Anfrageauswertung mit dem Luegen-Zensor fuer offene(!) Anfragen 
	 *  bei bekannten potentiellen Geheimnissen und einer als vollstaendig angenommenen Datenbankinstanz.
	 *  @param interaction Die an die Datenbank gestellte offene Anfrage mit genau einer freien Variable
	 *  
	 */
	@Override
	protected List<Formula> run(Client client, Formula interaction) throws DatabaseException, ProverException, UnsupportedFormulaException, DictionaryTooSmallException {		
		// Referenz auf LyingCensor für geschlossene Anfragen erstellen, falls noch nicht geschehen
		if( closedLyingCensor == null )
				closedLyingCensor = (LyingCensor)this.getServer().getCensorManagement().getCensor("Lying");
		
		// keine offene Anfrage?
		if ( ! interaction.isOpenFormula() )
			throw new UnsupportedFormulaException(interaction.toString() + " ist keine offene Anfrage");
		
		// freie Variablen ermitteln
		Set<String> freeVars = interaction.getFreeVariables();
		
		// benoetigtes Dictionary laden
		Map<String, List<String>> dictionary = this.fillDictionary(client, interaction, freeVars);
		
		List<Formula> result = new LinkedList<Formula>();
		// log und PotSec-Menge aus dem User-Objekt laden
		List<ConfidentialityPolicyItem> confidentialityPolicy = client.getUser().getConfidentialityPolicy().getCopyAsList();
		List<IdAndFormula> log = client.getUser().getKnowledge().getCopyAsList();		// virtuelles Wissen, wird zwischenzeitlich erweitert
		
		// Menge der Substitutionen, die im endgueltigen Completenesssentence enthalten sind
		// Diese ist noetig, weil der finale CS auch alle negierten Antworten abdecken soll, es
		// muessen somit nur die Substitutionen positiver Antworten ausgeschlossen werden
		Set<Map<String, String>> finalCompletenessSentenceSubstitutions = new HashSet<Map<String,String>>();
		
		// Phase 1 [enumerate the domain up to round k such that c_k is the last constant that makes phi(x) true]
		// k = Min{ i | Complete(phi(x), i) is true in db }
		
		int k = getLastConstantSatisfyingPhi(client, dictionary, interaction);
		logger.debug("OpenlyingCensor: Phase 1: k=" + k);
		
		// phi(c_j) als geschlossene Anfrage behandeln und Antwort des normalen Lying-Censors in Log und Antwortmenge uebernehmen
		Map<String, String> mapping;
		for( int j=1; j<=k; ++j ) {
			mapping = this.getFreeVarMapping(dictionary, freeVars, j);
			Formula closedQuery = new Formula(interaction);
			closedQuery.substituteVariables( mapping );
			Formula censoredAnswer = closedLyingCensor.run( client, new Formula(closedQuery), log).get(0);
			if( censoredAnswer.equals(closedQuery) ) {
				// positive Antwort (auch wenn gelogen) in Result aufnehmen
				result.add( censoredAnswer );
				// Substitution in finalem Completenesssentence ausschliessen
				finalCompletenessSentenceSubstitutions.add( mapping );
			}
			else {
				// negierte Antwort (evtl. gelogen) wird spaeter ueber den CompletenessSentence abgedeckt
			}
			log.add( new IdAndFormula(0, censoredAnswer) );
		}
		
		
		// Phase 2 [determine an 'enumeration excess' for exiting:
		//          if needed, further enumerate the domain to find suitable lies for 
		//          exiting with a true completeness statement]
		
		CompletenessSentenceFormula completenessSentence = this.generateCompletenessSentence( client, dictionary, interaction, k );
		Formula censoredAnswer = closedLyingCensor.run( client, new CompletenessSentenceFormula(completenessSentence), log ).get(0);
		// FIXME equals auf CompletenessSentence-Objekten liefert immer true, da nur Substitutionen verglichen werden
		// der LyingCensor modifiziert aber den Syntaxbaum des CompletenessSentence verbotener weise, sodass das gesamte
		// Objekt inkonsistent wird.
		//if( censoredAnswer.equals( completenessSentence ) ) {
		if( censoredAnswer.getRootChild().equals( completenessSentence.getRootChild() ) ) {
			// correct answer permitted
			
			logger.debug("OpenlyingCensor: Phase 2: log vereinigt completenessSentence NOT IMP pot_sec_disj ---> correct answer permitted");
			
			// das Entfernen von negativen Antworten entfaellt, da diese von vornherein nicht mit ins
			// Result aufgenommen wurden, sondern spaeter ueber den CompletenessSentence mit abgedeckt werden
			
			// der CompletenessSentence wird nach dieser IF-ELSE-Anweisung zum Result hinzugefuegt	
		}
		else {
			// lied answer requested
			
			logger.debug("OpenlyingCensor: Phase 2: log vereinigt completenessSentence IMP pot_sec_disj ---> lied answer requested");
			
			// determine a nonempty 'enumeration excess'
			
			// enthaelt nach Bestimmung des enum excess alle positiven Antworten (neg. Antworten sind nicht relevant, da diese
			// ueber den CompletenessSentence abgedeckt werden
			List<Formula> posAnswers = new LinkedList<Formula>();
			// enthaelt nach Bestimmung des enum excess die zu den positiven Antworten korrespondierenden Substitutionen der freien Variablen.
			// Diese Substitutionen werden spaeter fuer die Bildung des CompletenessSentence benoetigt.
			List<Map<String, String>> posSubstitutions = new LinkedList<Map<String,String>>();
			//this.getEnumerationExcessHeuristic( client, dictionary, interaction, k, log, confidentialityPolicy, posAnswers, posSubstitutions );
			this.getEnumerationExcess( client, dictionary, interaction, k, log, confidentialityPolicy, posAnswers, posSubstitutions );
			
			// fuege positiven Teil des enumeration excess ins Result ein und schliesse die korrespondierenden Substitutionen im finalen Completenesssentence aus
			result.addAll( posAnswers );
			finalCompletenessSentenceSubstitutions.addAll( posSubstitutions );
		}
		
		// finalen CompletenessSentence aufbauen und zum Ergebnis hinzufuegen
		CompletenessSentenceFormula finalCompletenessSentence = new CompletenessSentenceFormula( interaction, finalCompletenessSentenceSubstitutions );
		result.add( finalCompletenessSentence );
		
		return result;
	}
	
	/**
	 * Berechnet einen nicht leeren enumeration excess ab Element k in der Diagonalisierungsreihenfolge.
	 * Gesucht werden Indizes k+1 bis l mit k<l und Literalzeichen (leer oder Negation), sodass
	 * log vereinigt { litSign_k+1 phi(c_k+1), litSign_k+1 phi(c_k+1), ... phi(c_l) }
	 *     vereinigt { Complete(phi(x), l) } NOT_IMPLIES pot_sec_disj
	 * Arbeitsweise (Betrachte Durchlauf i):
	 * 		Es werden alle moeglichen Kombinationen betrachtet, wie die Literalzeichen von k+1 bis k+i
	 * 		gewaehlt werden koennen. Zuerst die, bei der nur k+i positiv gewaehlt wird, anschliessend
	 * 		diejenige, bei der k+i und k+1 positiv gewaehlt werden usw.
	 * Allgemein:
	 * 		Anz. Elemente im enumeration Excess in Durchlauf i: |_ log i _| + 1
	 * 		Wert von Element j in Durchlauf i ist positiv, wenn |_ i / 2^(j-1) _| mod 2 == 1, sonst negiert
	 * 
	 * Die Rueckgabe erfolgt ueber die Parameter posAnswers und posSubstitutions.
	 * Negative Antworten werden bei der Rueckgabe nicht betrachtet, da diese spaeter ueber den CompletenessSentence
	 * implizit erfolgen.
	 * 
	 * @param interaction der Query bzgl. dem der enumeration excess berechnet werden soll 
	 * @param k           Element in der Diagonalisierungsreihenfolge, ab dem der enumeration excess gesucht werden soll
	 * @param log         log
	 * @param pot_sec     pot_sec
	 * @param posAnswers  Formelmenge, die nach dem Finden des enumeration excess die positiven Antworten enthaelt
	 * @param posSubstitutions Vekor aus Maps, der nach dem Finden des enumeration excess die zu den positiven Antworten 
	 * 						   gehoerenden Substitutionen der freien Variablen enthaelt
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 * @throws ProverException
	 * @throws DictionaryTooSmallException
	 */
	private void getEnumerationExcess( Client client, Map<String, List<String>> dictionary, Formula interaction, int k, List<IdAndFormula> log, List<ConfidentialityPolicyItem> pot_sec, List<Formula> posAnswers, List<Map<String, String>> posSubstitutions ) 
								throws UnsupportedFormulaException, DatabaseException, ProverException, DictionaryTooSmallException {
		
		// FIXME kombinatorisches Problem mit exponentieller Laufzeit in der Groesse des Dictionaries
		//       Eine effizientere Loesung waere wuenschenswert.
		
		Boolean enumerationExcessFound = false;
		Set<String> freeVars = interaction.getFreeVariables();
		List<Formula> combinationComplete = new LinkedList<Formula>();
		CompletenessSentenceFormula completenessSentence;
		Map<String, String> mapping;
		List<Formula> tmpPosAnswers = new LinkedList<Formula>();
		List<Map<String, String>> tmpPosSubstitutions = new LinkedList<Map<String,String>>();
		
		int index = 1;
		while( ! enumerationExcessFound ) {
			// probiere naechste Kombination
			int elementCount = (int)Math.floor( Math.log(index) / Math.log(2) + 1);
			
			// resette temporaere pos. Antworten und zugehoerige Substitutionen sowie die temporaere Formelmenge combinationComplete
			// fuer neuen Versuch einen passenden enumeration excess zu finden
			tmpPosAnswers.clear();
			tmpPosSubstitutions.clear();
			combinationComplete.clear();
			
			// aktuelle Kombination in enumExcess berechnen
			for( int i=1; i<=elementCount; ++i) {
				mapping = this.getFreeVarMapping( dictionary, freeVars, k+i );
				Formula closedQuery = new Formula( interaction );
				closedQuery.substituteVariables( mapping );
				// Query negieren, wenn dies die aktuelle Kombination vorsieht
				if( Math.floor(index / Math.pow(2, i-1)) % 2 == 0 ) {
					closedQuery.negate();
				}
				else {
					// pos. Antwort in Menge der pos. Antworten und zugehoerige Substitutionen aufnehmen
					tmpPosSubstitutions.add( mapping );
					tmpPosAnswers.add( closedQuery );
				}
				combinationComplete.add( closedQuery );
			}
			// Completenesssentence aufnehmen
			completenessSentence = this.generateCompletenessSentence( client, dictionary, interaction, k+elementCount );
			combinationComplete.add( completenessSentence );
			
			// Implikation mit pot_sec_disj pruefen
			if( ! this.isConfidentialityPolicyViolated( combinationComplete, log, pot_sec ) ) {
				// 'nonempty enumeration excess' gefunden
				enumerationExcessFound = true;
			}
			else {
				++index;
			}
		}
		
		// gefundene positiven Antworten und zugehoerige Substitutionen eines gefundenen enum excess fuer
		// die Rueckgabe verarbeiten
		posAnswers.addAll( tmpPosAnswers );
		posSubstitutions.addAll( tmpPosSubstitutions );
	}
	
	
	/**
	 * Fuer die Formelmenge log u querys wird geprueft, ob durch diese die Disjunktion der potentiellen Geheimnisse
	 * aus der Sicherheitspolitik impliziert wird.
	 * @param querys Eine Menge von Querys, fuer die geprueft werden soll, ob sie zusammen mit dem Log die confidentiality policy verletzen
	 * @param log Das Vorwissen des Benutzers
	 * @param confidentialityPolicy Die Sicherheitspolitik in Form von potentiellen Geheimnissen
	 * @return true, falls durch log u querys die Disjunktion der potentiellen Geheimnisse impliziert wird; ansonsten wird false zurueck gegeben
	 * @throws ProverException Wird geworfen, falls im Rahmen des Theorembeweiser-Aufrufs ein Fehler auftritt
	 */
	protected boolean isConfidentialityPolicyViolated( List<Formula> querys, List<IdAndFormula> log, List<ConfidentialityPolicyItem> confidentialityPolicy ) throws ProverException {
		return closedLyingCensor.isConfidentialityPolicyViolated(querys, log, confidentialityPolicy);
	}
	
	
	/**
	 * Berechnet einen nicht leeren enumeration excess ab Element k in der Diagonalisierungsreihenfolge.
	 * Gesucht werden Indizes k+1 bis l mit k<l und Literalzeichen (leer oder Negation), sodass
	 * log vereinigt { litSign_k+1 phi(c_k+1), litSign_k+1 phi(c_k+1), ... phi(c_l) }
	 *     vereinigt { Complete(phi(x), l) } NOT_IMPLIES pot_sec_disj
	 * Die Rueckgabe erfolgt ueber die Parameter posAnswers und posSubstitutions.
	 * Negative Antworten werden bei der Rueckgabe nicht betrachtet, da diese spaeter ueber den CompletenessSentence
	 * implizit erfolgen.
	 * 
	 * @param interaction der Query bzgl. dem der enumeration excess berechnet werden soll 
	 * @param k           Element in der Diagonalisierungsreihenfolge, ab dem der enumeration excess gesucht werden soll
	 * @param log         log
	 * @param pot_sec     pot_sec
	 * @param posAnswers  Formelmenge, die nach dem Finden des enumeration excess die positiven Antworten enthaelt
	 * @param posSubstitutions Vekor aus Maps, der nach dem Finden des enumeration excess die zu den positiven Antworten 
	 * 						   gehoerenden Substitutionen der freien Variablen enthaelt
	 * @throws UnsupportedFormulaException
	 * @throws DatabaseException
	 * @throws ProverException
	 * @throws DictionaryTooSmallException
	 */
	private void getEnumerationExcessAlternative( Client client, Map<String, List<String>> dictionary, Formula interaction, int k, List<IdAndFormula> log, List<ConfidentialityPolicyItem> pot_sec, List<Formula> posAnswers, List<Map<String, String>> posSubstitutions ) 
								throws UnsupportedFormulaException, DatabaseException, ProverException, DictionaryTooSmallException {
		
		Set<String> freeVars = interaction.getFreeVariables();
		
		// ermittle alle Konstenten, die in der Anfrage, Log oder in der Policy vorkommenn
		Set<String> constantsOccuringInInteractionOrLogOrPolicy = new HashSet<String>();
		constantsOccuringInInteractionOrLogOrPolicy.addAll( interaction.getConstants() );
		for( IdAndFormula logEntry : log ) {
			constantsOccuringInInteractionOrLogOrPolicy.addAll( logEntry.getFormula().getConstants() );
		}
		for( ConfidentialityPolicyItem confPolItem : pot_sec ) {
			constantsOccuringInInteractionOrLogOrPolicy.addAll( confPolItem.getFormula().getConstants() );
		}
		
		// filtere alle Konstanten heraus, die auch im aktuellen Dictionary vorkommen
		Map<String, Set<String>> reducedDictionary = new HashMap<String, Set<String>>();
		for( Map.Entry<String, List<String>> dict : dictionary.entrySet() ) {
			reducedDictionary.put(dict.getKey(), new HashSet<String>(dict.getValue()));
			reducedDictionary.get(dict.getKey()).retainAll(constantsOccuringInInteractionOrLogOrPolicy);
		}
		
		// fuege Konstanten der Kombinationen k+1 bis k+3 hinzu um leere Dictionaries auszuschliessen
		for( int i=1; i<=3; ++i ) {
			Map<String, String> mapping = this.getFreeVarMapping(dictionary, freeVars, k+i);
			for( Map.Entry<String, String> substitution : mapping.entrySet() ) {
				reducedDictionary.get( substitution.getKey() ).add( substitution.getValue() );
			}
		}
		
		// alle Kombinationen des reducedDictionarys auflisten
		// Iteratoren initialisieren
		SortedMap<String, Iterator<String>> iterators = new TreeMap<String, Iterator<String>>();
		Map<String, String> iteratorValues = new HashMap<String, String>();
		for( Map.Entry<String, Set<String>> entry : reducedDictionary.entrySet() ) {
			Iterator<String> iterator = entry.getValue().iterator();
			iterators.put( entry.getKey(), iterator );
			iteratorValues.put( entry.getKey(), iterator.next() );
		}
		
		List<Map<String, String>> combinations = new ArrayList<Map<String, String>>();
		while( iterators.get(iterators.lastKey()).hasNext() ) {
			// aktuelle Kombination in Kombinationsliste aufnehmen
			combinations.add( new HashMap<String, String>(iteratorValues) );
			
			// ersten Iterator hochzaehlen (falls nicht mehr moeglich zuruecksetzen und folgenden Iterator hochzaehlen usw.)
			for( Map.Entry<String, Iterator<String>> entry : iterators.entrySet() ) {
				if( entry.getValue().hasNext() ) {
					// Iterator ein weiter setzen und diesen Wert merken
					iteratorValues.put( entry.getKey(), entry.getValue().next() );
					break;
				}
				else {
					if( entry.getKey().equals(iterators.lastKey()) ) {
						// fertig, alle Kombinationen wurden durchprobiert
						break;
					}
					// Iterator zuruecksetzen und Wert merken
					Iterator<String> iterator = reducedDictionary.get(entry.getKey()).iterator();
					iterators.put( entry.getKey(), iterator );
					iteratorValues.put( entry.getKey(), iterator.next() );
				}
			}
		}

		// Suche enumeration excess innerhalb der oben ermittelten Kombinationen, die ueberwiegend aus Konstanten bestehen,
		// die bereits im Log oder PotSec vorkommen
		Boolean enumerationExcessFound = false;
		List<Formula> combinationComplete = new LinkedList<Formula>();
		CompletenessSentenceFormula completenessSentence;
		Set<Map<String, String>> additionalCompletenessSentenceSubstitutions;
		Map<String, String> mapping;
		List<Formula> tmpPosAnswers = new LinkedList<Formula>();
		List<Map<String, String>> tmpPosSubstitutions = new LinkedList<Map<String,String>>();
		
		int index = 1;
		while( ! enumerationExcessFound ) {
			// probiere naechste Kombination
			int elementCount = (int)Math.floor( Math.log(index) / Math.log(2) + 1);
			
			if( elementCount > combinations.size() ) {
				// kein enumeration excess innerhalb der Kombinationen gefunden
				// -> Fallback auf das Standard-Verfahren
				
				this.getEnumerationExcess(client, dictionary, interaction, k, log, pot_sec, posAnswers, posSubstitutions);
				return;
			}
			
			// resette temporaere pos. Antworten und zugehoerige Substitutionen sowie die temporaere Formelmenge combinationComplete
			// fuer neuen Versuch einen passenden enumeration excess zu finden
			tmpPosAnswers.clear();
			tmpPosSubstitutions.clear();
			combinationComplete.clear();
			
			additionalCompletenessSentenceSubstitutions = new HashSet<Map<String,String>>();
			
			// aktuelle Kombination in enumExcess berechnen
			for( int i=0; i<elementCount; ++i) {
				mapping = combinations.get(i);
				Formula closedQuery = new Formula( interaction );
				closedQuery.substituteVariables( mapping );
				// Query negieren, wenn dies die aktuelle Kombination vorsieht
				if( Math.floor(index / Math.pow(2, i)) % 2 == 0 ) {
					closedQuery.negate();
				}
				else {
					// pos. Antwort in Menge der pos. Antworten und zugehoerige Substitutionen aufnehmen
					tmpPosSubstitutions.add( mapping );
					tmpPosAnswers.add( closedQuery );
				}
				combinationComplete.add( closedQuery );
				additionalCompletenessSentenceSubstitutions.add( mapping );
			}
			logger.debug("Teste Kombination: " + combinationComplete.toString());
			
			// Completenesssentence aufnehmen
			completenessSentence = this.generateCompletenessSentence( client, dictionary, interaction, k );
			completenessSentence.addSubstitutions( additionalCompletenessSentenceSubstitutions );
			combinationComplete.add( completenessSentence );
			
			// Implikation mit pot_sec_disj pruefen
			if( ! this.isConfidentialityPolicyViolated( combinationComplete, log, pot_sec ) ) {
				// 'nonempty enumeration excess' gefunden
				enumerationExcessFound = true;
			}
			else {
				++index;
			}
		}
		
		// gefundene positiven Antworten und zugehoerige Substitutionen eines gefundenen enum excess fuer
		// die Rueckgabe verarbeiten
		posAnswers.addAll( tmpPosAnswers );
		posSubstitutions.addAll( tmpPosSubstitutions );
	}
	
	
	
	
	
	
	
	private void getEnumerationExcessHeuristic( Client client, Map<String, List<String>> dictionary, Formula interaction, int k, List<IdAndFormula> log, List<ConfidentialityPolicyItem> pot_sec, List<Formula> posAnswers, List<Map<String, String>> posSubstitutions ) throws UnsupportedFormulaException, DatabaseException, ProverException, DictionaryTooSmallException {
		Random random = new Random();
		random.setSeed(12256454545L);
		
		Set<String> freeVars = interaction.getFreeVariables();
		
		List<Formula> tmpPosAnswers = new LinkedList<Formula>();
		Set<Map<String, String>> tmpPosSubstitutions = new HashSet<Map<String,String>>();
		
		boolean enumerationExcessFound = false;
		int iterationCounter = 1;
		int size = 1;
		while( ! enumerationExcessFound ) {
			// Groesse des EnumerationExcess bestimmen
			if( iterationCounter%10 == 0 )
				size = size * 2;
			
			// zufaellige Kombinationen (size viele) aus Dictionary bilden und fuer den EnumerationExcess positiv machen
			Set<Map<String, String>> combinations = new HashSet<Map<String, String>>();
			for( int i=0; i<size; ++i ) {
				// fuer jede zu bildende Kombination...
				Map<String, String> combination = new HashMap<String, String>();
				for( String freeVar : freeVars ) {
					// fuer jede fuer die Kombination zu betrachtende freie Variable...
					List<String> freeVarDict = dictionary.get(freeVar);
					combination.put(freeVar, freeVarDict.get(random.nextInt(freeVarDict.size())));
				}
				// FIXME: Kombinationen ausschliessen, die in der Diagonalisierungsreihenfolge an einer Position <= k vorkommen
				combinations.add(combination);
			}
			
			// EnumerationExcess aus Kombinationen aufbauen und testen, ob dieser akzeptiert werden darf
			// resette temporaere pos. Antworten und zugehoerige Substitutionen sowie die temporaere Formelmenge combinationComplete
			List<Formula> posFormulasAndCS = new LinkedList<Formula>();
			tmpPosAnswers.clear();
			tmpPosSubstitutions.clear();
			
			for( Map<String, String> combination : combinations ) {
				Formula closedQuery = new Formula( interaction );
				closedQuery.substituteVariables( combination );
				// pos. Antwort in Menge der pos. Antworten und zugehoerige Substitutionen aufnehmen
				tmpPosSubstitutions.add( combination );
				tmpPosAnswers.add( closedQuery );
				posFormulasAndCS.add( closedQuery );
			}
			
			// Completenesssentence generieren und zur Formelmenge hinzufuegen
			CompletenessSentenceFormula completenessSentence = this.generateCompletenessSentence( client, dictionary, interaction, k );
			completenessSentence.addSubstitutions( tmpPosSubstitutions );
			posFormulasAndCS.add( completenessSentence );
			
			// Implikation mit pot_sec_disj pruefen
			if( ! this.isConfidentialityPolicyViolated( posFormulasAndCS, log, pot_sec ) ) {
				// 'nonempty enumeration excess' gefunden
				enumerationExcessFound = true;
			}
			
			++iterationCounter;
		}
		
		// EnumerationExcess zurueckgeben: positive Antworten und Substitutionen fuer den CompletenessSentence
		posAnswers.addAll( tmpPosAnswers );
		posSubstitutions.addAll( tmpPosSubstitutions );
	}
	
	
	private void getEnumerationExcessProofHeuristik( Client client, Map<String, List<String>> dictionary, Formula interaction, int k, List<IdAndFormula> log, List<ConfidentialityPolicyItem> pot_sec, List<Formula> posAnswers, List<Map<String, String>> posSubstitutions ) throws UnsupportedFormulaException, DatabaseException, ProverException, DictionaryTooSmallException {
//		TheoremProver tp = new ProverNine();
//		
//		List<Formula> logFormulas = new LinkedList<Formula>();
//		for( IdAndFormula logFormula : log )
//			logFormulas.add(logFormula.getFormula());
//		
//		// teste ersten moeglichen CS
//		List<Formula> assumptions = new LinkedList<Formula>(logFormulas);
//		CompletenessSentenceFormula completenessSentence = this.generateCompletenessSentence( client, dictionary, interaction, k );
//		assumptions.add(completenessSentence);
//		ProverResult proverResultWithPotSec = tp.proveWithResult(assumptions, closedLyingCensor.createDisjunction(pot_sec), true);
//		if( proverResultWithPotSec.getResult() ) {
//			ProverResult proverResultWithoutPotSec = tp.proveWithResult(assumptions, new LinkedList<Formula>(), true).get(0);
//			if( proverResultWithoutPotSec.getResult() ) {
//				// Fall 1: CS erzeugt einen Widerspruch im Log
//				
//				// FIXME implementieren
//			} else {
//				// Fall 2: CS verletzt die confidentiality policy (Log ist dabei jedoch konsistent)
//				
//				// FIXME implementieren
//				
//				// Proof untersuchen und versuchen die Ursache fuer die Verletzung von pot_sec zu ermitteln
//				String proof = proverResultWithPotSec.getProof();
//			}
//		} else {
//			// CS ist harmlos -> fertig
//			
//		}
//		
//		
//		
//		
//		for( int i=1; i<=elementCount; ++i) {
//			mapping = this.getFreeVarMapping( dictionary, freeVars, k+i );
//			Formula closedQuery = new Formula( interaction );
//			closedQuery.substituteVariables( mapping );
//			// Query negieren, wenn dies die aktuelle Kombination vorsieht
//			if( Math.floor(index / Math.pow(2, i-1)) % 2 == 0 ) {
//				closedQuery.negate();
//			}
//			else {
//				// pos. Antwort in Menge der pos. Antworten und zugehoerige Substitutionen aufnehmen
//				tmpPosSubstitutions.add( mapping );
//				tmpPosAnswers.add( closedQuery );
//			}
//			combinationComplete.add( closedQuery );
//		}
//		// Completenesssentence aufnehmen
//		
//		combinationComplete.add( completenessSentence );
	}
}