package server.censor;

import java.util.List;

import notification.serverToClient.UpdateResultNotification;
import exception.DatabaseException;
import exception.ParserException;
import exception.ProverException;
import exception.UnsupportedConfidentialityPolicyException;
import exception.UnsupportedFormulaException;
import server.core.Client;
import server.core.Server;
import server.parser.Formula;
import server.parser.node.LiteralSequenceNode;

public abstract class UpdateCensor extends Censor {

	public UpdateCensor(String censorName, Server server) {
		super(censorName, server);
	}
	
	/**
	 * Teilt die Arbeit der Update-Zensoren auf die vier abstrakten Methoden vup(), vtr(), pup() und ptr() auf.
	 * 
	 * @param client
	 * @param interaction Eingabe des Benutzers. Kann ein VUP, VTR, PUP oder PTR sein. Wird auf die Methoden entsprechenden Methoden aufgeteilt.
	 * @return Ergebnismeldung des Update-Zensor.
	 * @throws Exception 
	 */
	public UpdateResultNotification update( Client client, Formula interaction ) throws Exception {
		switch ( interaction.getInteractionType() ) {
			case VIEW_UPDATE:
				return this.vup( client, interaction );
			case VIEW_UPDATE_TRANSACTION:
				return this.vtr( client, ((LiteralSequenceNode)interaction.getRootChild()).getLiteralList() );
			case PROVIDER_UPDATE:
				return this.pup( client, interaction );
			case PROVIDER_UPDATE_TRANSACTION:
				return this.ptr( client, ((LiteralSequenceNode)interaction.getRootChild()).getLiteralList() );
		}
		
		throw new UnsupportedFormulaException( interaction.toString() );
	}
	
	/**
	 * View Update mit nur einem einzigen Literal.
	 * 
	 * @param client
	 * @param literal Der Wert des Literals soll in der Sicht des Benutzers festgelegt werden.
	 * @return Ergebnismeldung als String.
	 */
	public abstract UpdateResultNotification vup( Client client, Formula literal );
	
	/**
	 * View Update mit einer Liste von Literalen, die in einer Transaktion aktualisiert werden sollen. 
	 * 
	 * @param client
	 * @param literals Die Werte der Literale aus der Liste sollen in der Sicht des Benutzers festgelegt werden.
	 * @return Ergebnismeldung als String.
	 * @throws DatabaseException 
	 * @throws UnsupportedFormulaException 
	 * @throws UnsupportedConfidentialityPolicyException 
	 * @throws ProverException 
	 * @throws ParserException 
	 * @throws Exception 
	 */
	public abstract UpdateResultNotification vtr( Client client, List<Formula> literals ) throws Exception;
	
	/**
	 * Provider Update (Admin Update) mit nur einem einzigen Literal.
	 * 
	 * @param client
	 * @param literal Der Wert des Literals soll in der Datenbankinstanz festgelegt werden.
	 * @return Ergebnismeldung als String.
	 */
	public abstract UpdateResultNotification pup( Client client, Formula literal );
	
	/**
	 * Provider Update (Admin Update) mit einer Liste von Literalen, die in einer Transaktion aktualisiert werden sollen.
	 * 
	 * @param client
	 * @param literals Die Werte der Literale aus der Liste sollen in der Datenbankinstanz festgelegt werden.
	 * @return Ergebnismeldung als String.
	 */
	public abstract UpdateResultNotification ptr( Client client, List<Formula> literals );

}
