package server.censor;

import java.util.LinkedList;
import java.util.List;

import server.parser.Formula;
import server.parser.node.Node;
import server.theoremprover.ProverNine;
import server.theoremprover.TheoremProver;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;
import exception.ProverException;
import exception.UnsupportedFormulaException;

public class CensorUtil {

	/**
	 * Konstruiert die Eingabe fuer den Theorembeweiser: Fuer die Formelmenge log u querys wird geprueft, ob durch diese eines der potentiellen Geheimnisse
	 * aus der Sicherheitspolitik impliziert wird.
	 * Hinweis: Diese Methode muss im Rahmen des RefusalCensor jeweils fuer log u querys und log u -querys aufgerufen werden.
	 * @param querys Eine Menge von Querys, fuer die geprueft werden soll, ob sie zusammen mit dem Log die confidentiality policy verletzen
	 * @param log Das Vorwissen des Benutzers
	 * @param confidentialityPolicy Die Sicherheitspolitik in Form von potentiellen Geheimnissen
	 * @return true, falls durch log u querys ein (beliebiges) potentielles Geheimnis impliziert wird; ansonsten wird false zurueck gegeben
	 * @throws ProverException Wird geworfen, falls im Rahmen des Theorembeweiser-Aufrufs ein Fehler auftritt
	 */
	public static boolean isConfidentialityPolicyViolated( List<Formula> querys, List<IdAndFormula> log, List<ConfidentialityPolicyItem> confidentialityPolicy ) throws ProverException {
//		if( confidentialityPolicy.isEmpty() ) {
//			// eine leere confidentiality policy wird trivialerweise nicht verletzt
//			return false;
//		}
		
		List<Formula> sos = new LinkedList<Formula>();
		
		// Benutzerwissen log Zeile fuer Zeile in Prover9-Eingabe schreiben
		for ( IdAndFormula logentry : log ) {
			sos.add( logentry.getFormula() );
		}
		
		// Query(s) hinzufuegen.
		sos.addAll( querys );
		
		// Pruefe, ob die Implikation log u query |= psi fuer ein psi aus pot_sec wahr ist
		TheoremProver tp = new ProverNine();
		
		List<Node> goals = new LinkedList<Node>();
		for( ConfidentialityPolicyItem policyItem : confidentialityPolicy ) {
			goals.add( policyItem.getFormula() );
		}
		
		return tp.prove( sos, goals, true );
	}
	
	/**
	 * Konstruiert die Eingabe fuer den Theorembeweiser: Fuer die Formelmenge log u query wird geprueft, ob durch diese eines der potentiellen Geheimnisse
	 * aus der Sicherheitspolitik impliziert wird.
	 * Hinweis: Diese Methode muss im Rahmen des RefusalCensor jeweils fuer log u query und log u -query aufgerufen werden.
	 * @param query Ein Query, fuer den geprueft werden soll, ob er zusammen mit dem Log die confidentiality policy verletzt
	 * @param log Das Vorwissen des Benutzers
	 * @param confidentialityPolicy Die Sicherheitspolitik in Form von potentiellen Geheimnissen
	 * @return true, falls durch log u query ein (beliebiges) potentielles Geheimnis impliziert wird; ansonsten wird false zurueck gegeben
	 * @throws ProverException Wird geworfen, falls im Rahmen des Theorembeweiser-Aufrufs ein Fehler auftritt
	 */
	public static boolean isConfidentialityPolicyViolated( Formula query, List<IdAndFormula> log, List<ConfidentialityPolicyItem> confidentialityPolicy ) throws ProverException {
		List<Formula> querys = new LinkedList<Formula>();
		querys.add( query );
		
		return CensorUtil.isConfidentialityPolicyViolated( querys, log, confidentialityPolicy );
	}
	
	/**
	 * Diese Methode realisiert die VariablenNegation auf einer Menge von Formeln, wie sie  
	 * Inference-proof View Update Transactions With Forwarded Refreshments in Definition 5
	 * definiert.
	 * Diese wird im ViewUpdateAlgorithmus dazu verwendet, moegliche Inferenzen zu berechnen.
	 * Sie behilft sich dazu der Methode neg(Formula) in der Klasse Node.
	 *  
	 * @params eine Menge von Formeln formulas und ein Literal, dieses muss vom Typ AtomPredicateNode sein
	 * @author benet
	 * @throws UnsupportedFormulaException 
	 */
	public static void variableNegationOnSetsOfFormulas(List<Formula> formulas, List<? extends Node> literals) throws UnsupportedFormulaException {
		for( Formula formula : formulas ){
			formula.neg( literals );
		}
	}
}
