package client.gui.usermanagement;

import java.io.IOException;
import java.util.List;
import java.util.Vector;

import javax.swing.table.AbstractTableModel;

import notification.serverToClient.ExceptionNotification;

import client.core.Client;
import exception.ParserException;

import server.parser.Formula;
import user.IdAndFormula;

public class UserInformationModel<T extends IdAndFormula> extends AbstractTableModel {
	private static final long serialVersionUID = 2724489715172101148L;
	
	private List<T> formulaList;
	private int identifier;
	private int userId;
	
	public UserInformationModel( int identifier, int userId ) {
		this( identifier, userId, new Vector<T>() );
	}
	
	public UserInformationModel( int identifier, int userId, List<T> list ) {
		this.formulaList = list;
		this.identifier = identifier;
		this.userId = userId;
	}

	@Override
	public int getColumnCount() {
		return 1;
	}

	@Override
	public int getRowCount() {
		if ( this.formulaList != null )
			return this.formulaList.size();
		else
			return 0;
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		if ( columnIndex == 0 && this.formulaList != null ) {
			return this.formulaList.get(rowIndex);
		} else {
			return null;
		}
	}
	
	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		// Nur Spalte 0 existiert
		if ( columnIndex != 0 || this.formulaList == null ) {
			return;
		}
		
		try {
			// Formel suchen
			T formula = this.getFormula( rowIndex );
			
			// String in formula umwandeln
			String newFormula = (String)aValue;
			
			if ( Client.GUI.editUserInformationEntry( this.userId, this.getIdentifier(), formula.getId(), newFormula) ) {
				formula.setFormula( new Formula(newFormula) );
			}
		} catch (IOException e) {
			Client.GUI.showError("Failed to edit user information.", e, ExceptionNotification.NONCRITICAL);
		} catch (ArrayIndexOutOfBoundsException e) {
			Client.GUI.showError("Unknown user information.", e, ExceptionNotification.NONCRITICAL);
		} catch (ParserException e) {
			Client.GUI.showError("Unable to parse formula. Shouldn't happen as the server should notice this in the first place.", e, ExceptionNotification.NONCRITICAL);
		}
	}

	public int getIdentifier() {
		return this.identifier;
	}
	
	public T getFormula(int index) {
		return this.formulaList.get(index);
	}
	
	/**
	 * Gibt den gespeicherten Vektor zurueck.
	 * @return Der gespeicherte Vektor.
	 */
	public List<T> getList() {
		return this.formulaList;
	}
	
	/**
	 * Setzt die gespeicherte Liste neu. Die Aenderung wird dem Benutzer angezeigt.
	 * @param list
	 */
	public void setList( List<T> list ) {
		this.formulaList = list;
		this.fireTableDataChanged();
	}
	
	public void addFormula( T formula ) {
		this.formulaList.add( formula );
    	this.fireTableRowsInserted( getRowCount()-1, getRowCount()-1 );
	}
	
    public void deleteFormula( T formula ) {
    	int index = this.formulaList.indexOf( formula );
    	if ( index >= 0 ) {
    		this.deleteFormula(index);
    	}
    }
	
	public void deleteFormula( int index ) {
		this.formulaList.remove(index);
		this.fireTableRowsDeleted( index, index );
	}

	/**
     * Direktes Manipulieren der Liste erlauben.
     * @return Immer true.
     */
	@Override
    public boolean isCellEditable(int row, int col) {
    	return true;
    }
	
	/**
	 * Wir wollen keine Tabellenkoepfe anzeigen, deshalb wird immer der leere String zurueckgegeben.
	 * @return Ein leerer String ("").
	 */
	@Override
    public String getColumnName(int column) {
		return "";
	}
}
