function [y1] = BQ70_RB50TS(x1)
% BQ70_RB50TS is neural network simulation function.
% It predicts TI BQ25570 efficiency according to the input voltage and
% current in addition to the storage voltage.
% 
% [y] = BQ70_RB50TS(x) takes these arguments as:
%   x = [Vin, Vstr, Ii]'
%       Voltage values in V (Volt)
%       Current value in uA (micro Ampere)
% and returns:
%   y = efficiency (in percent)
% 
% This model has a single hidden layer including 50 neurons with radial basis transfer function 
% and  a single neuron tangent sigmoid transfer function output layer.

% Coperight: Mojtaba Masoudinejad, 2018, Dortmund
% mojtaba.masoudinejad@tu-dortmund.de

% ===== NEURAL NETWORK CONSTANTS =====

% Input 1
x1_step1.xoffset = [0.328;1.5999;0.5];
x1_step1.gain = [0.447727781508843;0.769201184569824;1.99005965203807e-05];
x1_step1.ymin = -1;

% Layer 1
b1 = [-85.161251891221198207;2.292202756662252483;-136.20513379148368927;-115.93779619486376475;15.346637049662577468;-3.1861419234729009275;-239.65028116829591909;128.2127535096434201;68.73886673778351053;-8.4660634776171708182;-721.37908596394868255;251.74144001137287319;-24.446346380130041354;93.783162462435285534;0.82981911545746744885;-559.24932836204391151;-605.99415010501593315;8.5949067943569339434;6.2941756942860607182;32.731073477445306708;-120.77176507509032888;-447.10984977466245027;-35.881031590204159443;400.92548085981184158;-0.4677501856984328521;396.70231801828094831;1.03463371141388083;43.036061965753972913;23.034276637368535035;-21.795207544369976205;-22.247770674003159996;58.850140511011836963;33.616223483101514091;-50.530431967941304094;45.064620007739726759;-64.870125055896735944;21.404644129652584894;0.11999601385065869485;16.537646609072083947;53.280721289945724095;800.89759331949392163;16.552710240373727402;21.396018506069907517;2.3882190250096786421;-34.221368339681838222;33.871149063012524039;-37.385482717911102668;-1433.8493327692669936;35.540013683799067223;-28.917119632723942146];
IW1_1 = [8.8099709951403646357 -3.9289043427657670549 -83.607165524045569782;-8.2549703553955691149 4.9877105460515709368 0.34408461956057856401;73.543697221980764311 -32.703685988631114867 -140.90914706111823307;8.991828460525541189 -34.275049458282218495 -97.239648830571240978;-8.9073974273341249841 7.181277408700559306 16.144857209155709654;7.0001391715812477656 -4.0931931336509217445 -0.71149021852949601019;45.786087373555027114 -20.802677392888870855 -247.05745983029763124;-21.47590166656985744 -36.255256676763245594 101.37735761093155418;-55.000142311679930174 -58.690393812799726447 17.470240295895305138;-4.1174546174171196 -6.8074392266457399714 0.5627571878278593287;-0.21221753169888546431 0.11660916015677330604 -719.09833596829764701;-515.24243905013429412 -168.63266071944602231 -33.298349840262197574;12.17601317266541372 -5.5167059541674952072 -25.612351219578105344;-8.8917351145554128777 3.9907331734060842443 92.267178448421404369;-0.40185057679618085835 0.055499757221332816914 -0.73458409882401387581;-0.033714380849978853316 0.058162991507459058105 -558.64440196526379623;0.018895893016269132286 0.03624784506048326499 -605.5452674094727854;4.2156151665986980603 6.8778419415791356428 -0.57459404426240645591;-38.274935963999212163 23.933697362344069859 0.20392755810644186409;-6.6713147168790358776 19.461684176205466201 22.05497279561806323;415.50026064330256759 6.4729223825331976272 -77.292413787695622318;-0.057511680247848619152 0.059194690945009689753 -446.91586837519452047;74.655402530941145756 -11.502220932237465334 -71.960967746448858406;-66.040549367906209 33.399791511160316304 412.9759342514500986;-0.66264889090239720293 -0.19215331423236917119 -0.91932368400295350064;0.050657901624514695138 -0.053170602664345138766 396.08062364682888301;-64.486969124552814492 35.157127576878821174 -3.4447594052431806944;-72.370728514083424443 336.1073631095181895 -73.068425950161412175;-9.6514397196950501012 7.6387765729206362764 24.206230357249157947;-28.844379316089376886 72.241417749092676104 -60.41965780171803857;-29.441191350905221213 73.785651430873258505 -61.723993858669011559;93.923029035245846785 48.22657425221105143 52.361465279416535168;-6.4081438933653966927 19.346572777177168945 22.945252247032215109;104.99227264122083625 -0.084165331668896115058 -133.29313287145109257;62.090935382866312864 -42.155673443718775673 56.48301095847797626;-92.141216527427459937 -41.276020948772760732 -32.243336529864606632;-39.748718500464633507 -3.3942172642480272771 54.996873007004751344;2.1886766861020792163 -0.5089070017956603964 -0.6932887628389186041;40.932032792223211004 -35.464294753847887876 1.2870663042380885432;15.883574392468204906 -69.832203712383105199 23.420710592062384592;1.0664533829000755194 -0.18448393893081857509 797.71664064732124189;40.969953170503814022 -35.513303387205347406 1.2797319182533979376;-39.716281982013356355 -3.3724439260365826065 54.957024569323330354;0.37999863588488880239 0.0064049762681098118544 0.038162792013508160804;6.3045650032739626312 -19.35352661683755926 -23.523466932659651718;23.799180453073091712 35.024215380887703475 -1.8612680543752153639;-26.416253536083008413 -38.505015149772830796 2.0980480900039455427;-0.26505228392196694687 0.12438431085355951267 -1431.2525231313670702;25.046687949095453263 36.670227731883258571 -1.9734328071770608215;-27.848305341498814869 -3.3032444279681705268 -15.036487171563608811];

% Layer 2
b2 = 2.2461377496881627103;
LW2_1 = [-16.284043308613213696 -0.56154535255472171063 0.050217907440066288094 -0.21052144764575544955 0.20113618593849261207 -0.49883620016614327319 0.051320922470721661024 0.024717535403935630284 -0.044074423688221189832 -6.1878713108030867218 457.8221174789537713 0.10581386335523471076 -0.13656531455600912084 14.980892299835989689 -1.0609795818753799956 145.3547726679817913 -48.498946253903099546 6.1272782798810698779 -0.31065779076230343181 17.437613315582115803 0.045040140876389724023 72.109559229695435079 -0.0085133937766138333136 -0.047893830133331781462 -0.46561949260732837796 -196.06050135197094164 0.088750535186966392209 -0.067064463857700290506 -0.21936666307405905396 -5.1108230836205645886 5.1271170464613469164 -0.030438830975316393673 -52.372190484323489557 0.042103817113172514741 0.24032751025654261468 0.086115120848873710768 -4.6739221345450348011 -0.15230417007781421601 4.4751747243946660149 0.038753716792714838768 -119.88649102647650579 -4.4717241104510474514 4.6919418392803864393 -66.122888388676514637 35.173333004106552835 -18.939701691486401813 -20.238942496290817985 -691.56463267147682927 39.104923471016235226 -0.03735769365079723292];

% Output 1
y1_step1.ymin = -1;
y1_step1.gain = 0.0200002226111087;
y1_step1.xoffset = 2.37595156899639e-05;

% ===== SIMULATION ========

% Dimensions
Q = size(x1,2); % samples

% Input 1
xp1 = mapminmax_apply(x1,x1_step1);

% Layer 1
a1 = radbas_apply(repmat(b1,1,Q) + IW1_1*xp1);

% Layer 2
a2 = tansig_apply(repmat(b2,1,Q) + LW2_1*a1);

% Output 1
y1 = mapminmax_reverse(a2,y1_step1);
end

% ===== MODULE FUNCTIONS ========

% Map Minimum and Maximum Input Processing Function
function y = mapminmax_apply(x,settings)
  y = bsxfun(@minus,x,settings.xoffset);
  y = bsxfun(@times,y,settings.gain);
  y = bsxfun(@plus,y,settings.ymin);
end

% Radial Basis Transfer Function
function a = radbas_apply(n,~)
  a = exp(-(n.*n));
end

% Sigmoid Symmetric Transfer Function
function a = tansig_apply(n,~)
  a = 2 ./ (1 + exp(-2*n)) - 1;
end

% Map Minimum and Maximum Output Reverse-Processing Function
function x = mapminmax_reverse(y,settings)
  x = bsxfun(@minus,y,settings.ymin);
  x = bsxfun(@rdivide,x,settings.gain);
  x = bsxfun(@plus,x,settings.xoffset);
end
